// The following messages contain the C code to my 6502 disassembler.
// Let me know if you find any bugs in it. To use it, just compile and
// run:
//      dasm65 filename.prg
// It assumes the first two bytes will be the start address (standard
// Commodore disk format).
//
//----------------------------------------------------------------------
//
// Jeff Hunsinger
// jeffh@oakhill-csic.sps.mot.com

/* 6502 Disassembler */
/* Copyright 1995, Jeff Hunsinger */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "dasm65.h"
#include "mnem65.c"

struct infodata {
	char    *label;
	short   addr;
	char    ltype;
	struct  infodata *next;
	};

typedef struct infodata node;

struct lbl {
	char label[12];
	short addr;
	};

typedef struct lbl labeltable;

void decode(unsigned short);
void prnt(unsigned short);

int getbyte(void);
short rel(char);
node *init_list(void);
void insert(short, const char);
node *find_node(short);
void wipe_list(void);
void buildlabel(void);
void printaddr(unsigned short);
void printlabel(unsigned short);
unsigned char asc2hex(char);
void readopts(int argc, char *argv[]);
short str2hex(char *);
char ishex(char);
char isascii(short);
unsigned char ascii2hex(char);

FILE *infile;
short pc, start_addr;
node *list;
char *fname;
char *lblname;
char options=0;

void main(int argc, char *argv[])
{
	int ch;

	if (argc < 2) {
	  printf("\nUsage: dasm65 [-v -s <start address> -l <label_file>] filename\n");
	  exit(0);
	}

	readopts(argc, argv);

	if ((infile=fopen(fname,"rb"))==NULL)  {
	  fprintf(stderr,"Error: file '%s' does not exist.\n",fname);
	  exit(0);
	}

	list = init_list();     /* Initialize address table */

	if (!(options & START)) {
	/* Read start address from file */
	  start_addr = getbyte();       /* Lo byte */
	  start_addr+= getbyte()*256;   /* Hi byte */
	}

	pc = start_addr-1;

	while((ch=getbyte())!=EOF) /* First pass - create label table */
		decode((unsigned short) ch);

	fclose(infile);

	buildlabel();           /* Read in address labels */
	infile = fopen(fname, "rb");

	if (!(options & START)) {
	  start_addr = getbyte();
	  start_addr+= getbyte()*256;
	}
	pc = start_addr-1;
	if (options & VERBOSE)
		printf("\t\t\t");
	printf("\tORG\t");
	printaddr(start_addr);
	printf("\n\n");

	while((ch=getbyte())!=EOF) {            /* Decode it */
		prnt((unsigned short) ch);
	}

	fclose(infile);
	free(fname);

	wipe_list();            /* Clean up memory */

}

void decode(unsigned short op)          /* Decode opcode */
{
  unsigned char op1,op2;

  switch(mnem[op].mode) {
	case REL:                               /* RELATIVE */
		op1=getbyte();                  /* Get relative offset */
		insert(rel(op1), mnem[op].addrtype);
		break;
	case ZPG:                               /* Zero Page */
	case ZPX:                               /* Zero Page, X */
	case ZPY:                               /* Zero Page, Y */
		op1=getbyte();                  /* Get zero page address */
		insert(op1, mnem[op].addrtype);
		break;
	case IMP:                               /* Implicit */
	case ACC:                               /* Accumalator */
		break;
	case IND:                               /* Indirect */
	case ABS:                               /* Absolute */
	case ABX:                               /* Absolute,X */
	case ABY:                               /* Absolute,Y */
		op1= getbyte();                 /* Get address */
		op2= getbyte();                 /* Get address */
		insert((op2*256+op1), mnem[op].addrtype);
		break;
	case INX:                               /* Indirect by X */
	case INY:                               /* Indirect by Y */
		op1= getbyte();                 /* Get address */
		insert((short) op1, mnem[op].addrtype);
		break;
	case IMM:                               /* Immediate */
		op1 = getbyte();
		break;
	default:
		break;
  }
}


void prnt(unsigned short op)            /* Decode opcode */
{
  unsigned char op1,op2;

  if (options & VERBOSE)
	  printf("%04x\t",pc);

  switch(mnem[op].mode) {
	case REL:                               /* Relative */
		op1=getbyte();
		if (options & VERBOSE)
		  printf("%02x %02x\t\t",op,op1);
		printlabel(pc-1);
		printf("\t%s\t",mnem[op].opn);
		printaddr(rel((char) op1));
		printf("\n");
		break;
	case ZPG:                               /* Zero Page */
		op1= getbyte();                 /* Get zero page address */
		if (options & VERBOSE)
		  printf("%02x %02x\t\t",op,op1);
		printlabel(pc-1);
		printf("\t%s\t",mnem[op].opn);
		printaddr(op1);
		printf("\n");
		break;
	case ZPX:                               /* Zero Page,X */
		op1= getbyte(); /* Get zero page address */
		if (options & VERBOSE)
		  printf("%02x %02x\t\t",op,op1);
		printlabel(pc-1);
		printf("\t%s\t",mnem[op].opn);
		printaddr(op1);
		printf(",X\n");
		break;
	case ZPY:                               /* Zero Page */
		op1= getbyte(); /* Get zero page address */
		if (options & VERBOSE)
		  printf("%02x %02x\t\t",op,op1);
		printlabel(pc-1);
		printf("\t%s\t",mnem[op].opn);
		printaddr(op1);
		printf(",Y\n");
		break;
	case ABX:
		op1= getbyte();
		op2= getbyte();
		if (options & VERBOSE)
		  printf("%02x %02x %02x\t",op,op1,op2);
		printlabel(pc-2);
		printf("\t%s\t",mnem[op].opn);
		printaddr(op2*256+op1);
		printf(",X\n");
		break;
	case ABY:
		op1= getbyte();
		op2= getbyte();
		if (options & VERBOSE)
		  printf("%02x %02x %02x\t",op,op1,op2);
		printlabel(pc-2);
		printf("\t%s\t",mnem[op].opn);
		printaddr(op2*256+op1);
		printf(",Y\n");
		break;
	case IND:
		op1= getbyte();
		op2= getbyte();
		if (options & VERBOSE)
		  printf("%02x %02x %02x\t",op,op1,op2);
		printlabel(pc-2);
		printf("\t%s\t(",mnem[op].opn);
		printaddr(op2*256+op1);
		printf(")\n");
		break;
	case IMP:                               /* Implicit */
	case ACC:
		if (options & VERBOSE)
		  printf("%02x\t\t",op1);
		printlabel(pc);
		printf("\t%s\n",mnem[op].opn);
		break;
	case INX:                               /* Indexed by X */
		op1= getbyte(); /* Get address */
		if (options & VERBOSE)
		  printf("%02x %02x\t\t",op,op1);
		printlabel(pc-1);
		printf("\t%s\t(",mnem[op].opn);
		printaddr(op1);
		printf(",X)\n");
		break;
	case INY:                               /* Indexed by Y */
		op1= getbyte(); /* Get address */
		if (options & VERBOSE)
		  printf("%02x %02x\t\t",op,op1);
		printlabel(pc-1);
		printf("\t%s\t(",mnem[op].opn);
		printaddr(op1);
		printf("),Y\n");
		break;
	case IMM:                               /* Immediate */
		op1= getbyte();
		if (options & VERBOSE)
		  printf("%02x %02x\t\t",op,op1);
		printlabel(pc-1);
		printf("\t%s\t#$%02x\n",mnem[op].opn,op1);
		break;
	case ABS:                               /* Absolute */
		op1= getbyte(); /* High byte of address */
		op2= getbyte(); /* Low byte of address */
		if (options & VERBOSE)
		  printf("%02x %02x %02x\t",op,op1,op2);
		printlabel(pc-2);
		printf("\t%s\t",mnem[op].opn);
		printaddr(op2*256+op1);
		printf("\n");
		break;
	default:                                /* Illegal */
		if (options & VERBOSE)
		    printf("%02x\t\t",op);
		printlabel(pc);         /* Print out address if necessary */
		printf("\tFCB\t$%02x\t\t; %c\n", op, isascii(op));
		break;
  }
}



short rel(char offset)
{
	return(offset+pc+1);
}

int getbyte(void)
{
	pc++;
	return(fgetc(infile));
}

char isascii(short ch)
{
	if ((ch>=' ') && (ch<='z'))
	  return(ch);
	else return('.');
}

node *init_list(void)
{
	node *head;

	head = (node *) malloc(sizeof(node));
	head->next = (node *) NULL;
	return(head);
}

/* Add node to structure */
/* Returns pointer to node. Expects address label as parameter */
void insert(short addr, const char addrtype)
{
	node *tmp, *prev, *p;

	prev = list;
	tmp = list->next;

	/* Find insertion point */
	while( (tmp != NULL) && (tmp->addr < addr) ) {
	  prev = tmp;
	  tmp = tmp->next;
	}

	/* Insert node */
	if ((tmp == NULL) || (tmp->addr != addr)) {  /* Insert only if new address */
		p=(node *) malloc(sizeof(node));
		if (p != (node *) NULL) { 
		  p->addr = addr;
		  p->next = tmp;
		  p->label = NULL;
		  p->ltype = addrtype;
		  prev->next = p;
		}
		else {
		  printf("Out of memory...\n");
		  exit(0);
		}

	}

}

/* Searches for address in list. Returns NULL if not found. */
node *find_node(short addr)
{
	node *tmp;

	tmp = list->next;
	while((tmp != NULL) && (tmp->addr != addr))
		tmp = tmp->next;
	return(tmp);
}

/* Print address only if referenced elsewhere. Print label if defined. */
void printaddr(unsigned short addr)
{
	node *p;

	if ((p=find_node(addr)) != NULL) {
		if (p->label != NULL)
			printf("%s",p->label);
		else {
		  if (addr<0x100)
			printf("D%02x",addr);
		  else  printf("%c%04x",p->ltype,addr);
		}
	}
	else    printf("$%04x",addr);
}

/* Print label if defined, otherwise print the address. */
void printlabel(unsigned short addr)
{
	if (find_node(addr) != NULL)
	  printaddr(addr);      /* Print out address if necessary */
}

/* Copy appropriate label definitions into address table */
void buildlabel(void)
{
	labeltable data;
	node *p;
	FILE *fp;
	char hibyte,lobyte;
	char dummy[5];

	if (options & LABEL)
	  if ((fp = fopen(lblname, "r")) != NULL) {
	    while(!feof(fp)) { 
		fscanf(fp,"%s\t%s\t$%c%c",data.label,dummy,&hibyte, &lobyte);
		data.addr = (asc2hex(hibyte)*256 + asc2hex(lobyte));
		if ((p=find_node(data.addr)) != NULL) {
			p->label = (char *)malloc(sizeof(char) * sizeof(data.label));
			strcpy(p->label, data.label);
		} 
	    } 
	    fclose(fp);
	  } 
}

unsigned char asc2hex(char num)
{
	if (num < 'A')
	  return(num-'0');
	else return(num-'A');
}

void wipe_list(void)
{
	node *nxt, *tmp;

	tmp = list;
	nxt = list->next;
	while(nxt != NULL) {
	  if (tmp->label != NULL)
		free(tmp->label); 
	  free(tmp);
	  tmp=nxt;
	  nxt=nxt->next;
	}
	free(tmp);
}

void readopts(int argc, char *argv[])
{
	int i;

	for(i=1;i<argc;i++)
		if (argv[i][0]=='-')
		  switch(argv[i][1]) {
			case 'v':
				options|=1;
				break;
			case 's':
				start_addr=str2hex(argv[++i]); 
				options|=2;
				break;
			case 'l':
				options|=4;
				lblname=argv[++i]; 
				break;
			default:
				printf("Invalid option.\n");
				printf("Usage: dasm08 [-v -l <lable file> -s <start address>] file.obj\n");
				exit(0);
		  }
		else fname=argv[i];
}

short str2hex(char *ptr)
{
	short num=0;
	char i=0;

	while((i<4) && (ishex(*(ptr+i))))
		num=num*16+(short) ascii2hex(*(ptr+i++));
	return(num);
}

char ishex(char ch)
{
	return(((ch >= '0') && (ch <= '9')) || ((ch>='a') && (ch<='f')));
}


unsigned char ascii2hex(char byte)
{
	if ((byte >= '0') && (byte <='9'))
		return(byte-'0');
	else return(byte-'a'+10);
}
