%
% Dither an image to create a 4-color texture file for C64 applications
%
% SLJ 1/25/2024
%

clear all;

%dtype = 1  % Floyd-Steinberg
%dtype = 2  % Sierra lite
%dtype = 3 % Stucki
dtype = 4 % Closest match

if false
files = {
 "A80.png"
 "B80.png"
 "C80.png"
 "D80.png"
 "E80.png"
 "F80.png"
 "G80.png"
 "H80.png"
 "I80.png"
 "J80.png"
 "K80.png"
 "L80.png"
 "M80.png"
 "N80.png"
 "O80.png"
 "P80.png"
 "Q80.png"
 "R80.png"
 "S80.png"
 "T80.png"
 "U80.png"
 "V80.png"
 "W80.png"
 "X80.png"
 "Y80.png"
 "Z80.png"
}
end
files = {
 "redbg.png"
}

for fcount=1:length(files)

fname = files{fcount};

data = imread(fname)*255;

if length(size(data))==2  % Grayscale image
  dither(:,:,1) = data;
  dither(:,:,2) = data;
  dither(:,:,3) = data;
else
  dither = data;
end
texture = zeros(size(data(:,:,1))); % This will be color number, rather than rgb

% r g and b values above are 0..255

if true
colors = [
        0 0 0     % black
        200 0 0   % red
        200 0 200 % violet
        200 200 200]; % lt gray
else
colors = [
        0 0 0 % black
        137 219 162 % cyan blue
        234 166 58   % orange (red)
        187 187 187 % lt gray
        ];
endif

if dtype==1
    % Floyd-Steiberg (1/16)
    %      X   7
    %  3   5   1
    dmat = [  % +row +col factor
      0 1 7/16
      1 -1 3/16
      0 -1 5/16
      1 1 1/16
      ];
elseif dtype==2
    % Sierra lite (1/4)
    %      X   2
    %  1   1
    dmat = [  % +row +col factor
      0 1 1/2
      1 -1 1/4
      0 -1 1/4
      ];
elseif dtype==3
      % Stucki
%             X   8   4
%     2   4   8   4   2
%     1   2   4   2   1
%
%           (1/42)
    dmat = [
      0 1 8/42
      0 2 4/42
      1 -2 2/42
      1 -1 4/42
      1 0 8/42
      1 1 4/42
      1 2 2/42
      2 -2 1/42
      2 -1 2/42
      2 0 4/42
      2 1 2/42
      2 2 1/42
    ];
elseif dtype==4 % closest match
    dmat = [0 0 0];
endif

% Now dither the image
rowmax = 160;
colmax = 80;
drow=0;
dcol=0;

for row=1:rowmax
  for col=1:colmax
    r = double(dither(row,col,1));
    g = double(dither(row,col,2));
    b = double(dither(row,col,3));

    % Find closest color

    mindist = 1e6;
    mincol = 0;
    for k=1:4
      d0 = (r-colors(k,1))^2 + (g-colors(k,2))^2 + (b-colors(k,3))^2;
      if d0 < mindist
       mincol=k;
       mindist = d0;
      endif
    endfor

    % Calculate error

    err_rgb = [
      r - colors(mincol,1)
      g - colors(mincol,2)
      b - colors(mincol,3)
      ];

    % And diffuse

    dither(row,col,:) = colors(mincol,:)';
    texture(row,col) = mincol-1; % 0..3
    for k=1:length(dmat(:,1))
      drow = row+dmat(k,1);
      dcol = col+dmat(k,2);
      c = dmat(k,3);
      if (drow <= rowmax) && (dcol <= colmax) && (dcol > 0)
        dither(drow,dcol,:) = squeeze(dither(drow,dcol,:)) + c*err_rgb;
      endif
    endfor

  endfor
endfor

% Display the results

figure(1); clf; imshow(data);
figure(1+dtype); clf; imshow(dither);


% Output a C64 bitmap, to check in VICE

k=0
bitmap = zeros(8*40*20);  # 320x200
for row=1:rowmax
  for col=1:colmax
    x = col-1;
    y = row-1;
    idx = 1 + bitand(y,248)*40 + bitand(y,7) + bitand(2*x,504);

    bitp = texture(row,col)*(4^(3-bitand(x,3)));
    bitmap(idx) = bitor(bitmap(idx),bitp);

    k=k+1;
    if (row==999)
      debug = [k x y idx bitmap(idx)]
    endif
  endfor
endfor

% Check to see if it worked

cmap = zeros(160,80);
idx = 1
y = 0
offset = 0
for row=1:160
  x=0;
  y=row-1;
  for col=1:40
    idx = 1 + bitand(y,248)*40 + bitand(y,7) + 8*(col-1);
    byte = bitmap(idx);
    x=x+1;
    cmap(row,x) = bitand(byte,0xc0)/64;
    x=x+1;
    cmap(row,x) = bitand(byte,0x30)/16;
    x=x+1;
    cmap(row,x) = bitand(byte,0x0c)/4;
    x=x+1;
    cmap(row,x) = bitand(byte,0x03);
  endfor
endfor
%figure(16); clf; surf(cmap);
%xlabel("x"); ylabel("y");

bitname = [fname(1:3) ".bmap"];
fid = fopen(bitname,"w");
fwrite(fid,0); % Put in a fake load address of $8000
fwrite(fid,128);
for k=1:length(bitmap)
  fwrite(fid,bitmap(k));
endfor
fclose(fid);

endfor

