c64fILE
raycast3.u.s








*
* A "16-bit" version, where the
* raycaster calculations are done
* in 16 bits rather than 8 bits,
* to improve some resolution and
* distance issues seen in the 8 bit
* version.
*
* SLJ 1/3/2024
*

********************************
*
* Arena -- a simple code to wander
* through a map, using the racasting
* libraries.
*
* SLJ 12/30/2023
*
********************************

********************************
*
* Arena
*
* An attempt at making a 3D raycasting
* engine. This code implements the
* routines from
* http://lodev.org/cgtutor/raycasting.html
*
* S Judd
* 11/2/2023
* (One more for old time's sake...)
*
********************************

;
; Memory map:
;   3000-3fff: library code
;   4000-4bff: Open (sprites etc.)
;   4c00-4fff: color map, buffer 1
;   5000-5fff: tables
;   6000-7fff: buffer 1
;   8000-8bff: Open (sprites etc.)
;   8c00-8fff: color map, buffer 2
;   9000-9fff: tables, misc
;   a000-bfff: buffer 2
;   c000-cfff: map
;
; Note that the render code bitmap
; size is 160x160.  Lines 161-200
; are left alone and are available
; for screen display (health, radar,
; etc.) by the user.
;


         org $3000

xmaptex equ $5000     ;map textures, x-direction
ymaptex equ $5080     ;map textures, y-direction
lo160 equ $5100       ;x*160 lo
hi160 equ $5200       ;x*160 hi
xtextab equ $5300     ;table of 80*x/256
invtablo equ $5400    ;65536/x
invtabhi equ $5500    ;
multlo1 equ $5600     ;f(a-b) 256 bytes
multlo2 equ $5700     ;f(a+b) 512 bytes
multhi1 equ $5900     ;high bytes same
multhi2 equ $5a00
invzlo equ $5c00      ;d/z, for GetHeight
invzhi equ $5d00
mapylo equ $5e00      ;low and high address
mapyhi equ $5e80      ;of map coords
sintablo equ $5f00    ;sin(x),cos(x)
costablo equ sintablo+16
sintabhi equ $5f80
costabhi equ sintabhi+16

Code1RowLo equ $9800  ;jump index
Code1RowHi equ $9900
Code2RowLo equ $9a00
Code2RowHi equ $9b00

Code3RowLo equ $9c00  ;jump index
Code3RowHi equ $9d00
Code4RowLo equ $9e00
Code4RowHi equ $9f00

black equ 0
white equ 1
red equ 2
cyan equ 3
purple equ 4
green equ 5
blue equ 6
yellow equ 7
orange equ 8
brown equ 9
ltred equ 10
gray1 equ 11
gray2 equ 12
ltgreen equ 13
ltblue equ 14
gray3 equ 15

;
; The render code uses several
; zero page locations which should
; be considered reserved
;
; 58-5f fast multiply pointers
; f9-fc misc (subroutine passing etc.)
; fd-ff texture pointer
;
  uint texp@$fd       ;24-bit pointer
zp equ $fb
ztemp equ $fb

zp1 equ $58           ;Fast multiply
zp2 equ $5a           ;pointers
zp3 equ $5c
zp4 equ $5e

;
; Table offsets
;

textab  equ $fc00     ;scaled texture index

Code1      equ $0000  ;bank 2 code loc
Code2      equ $4000  ;x>128
Code3      equ $8000  ;bank 2 code loc
Code4      equ $c000  ;x>128


*-------------------------------
*
* Jump table
*
*-------------------------------

; jmp TestCode

         jmp InitGfx
         jmp AddTexture
         jmp InitBitmaps
         jmp ClearBitmap
         jmp SetVICBuffer
         jmp DrawCamera
         jmp RayCast

********************************
*
* Draw the scene: iterate over the
* camera plane, cast rays, and draw
*
* theta - look angle (0..63)
* xpos,ypos - position in the map
*           - high byte = integer
*           - low byte = fraction
*
* We use two rays, one to the left of
* the center line (neg) and one to
* the right of the center line (pos).
*
* We then iterate over the camera
* plane by adding a deltax or deltay
* to the neg and pos ray directions
* until the entire plane is covered.
*
* The rays are computed as 16-bit
* fixed-point numbers.
*
********************************

sub DrawCamera(ubyte theta@$f7, int x0@$f8, int y0@$fa, ubyte buf@$fc)

  int xpos,ypos
  ubyte theta0,buffer
  int dirx,diry
  int rxplus,ryplus,rxminus,ryminus
  int DX128, DY128

  byte count
  byte dx,dy,quadrant
  ubyte xp,xm
  byte x1,y1,y2
  ubyte texture

  xpos = x0
  ypos = y0
  buffer = buf

; Get look ray direction
lookdir
    ldx theta
    lda costablo,x    ;63*cos(theta)
    sta dirx
    lda costabhi,x
    sta dirx+1

    lda sintablo,x
    sta diry
    lda sintabhi,x
    sta diry+1

; println
; println "dirx=",dirx," diry=",diry

 do 0
         ldx #00
         lda diry
         asl
         lda diry+1
         rol
         sta DY128
         bcc :c1
         dex
:c1      stx DY128+1

         ldx #00
         lda dirx
         asl
         lda dirx+1
         rol
         sta DX128
         bcc :c2
         dex
:c2      stx DX128+1
 fin

         lda diry
         sta dy128
         lda diry+1
         ldx #6
:l1      cmp #$80
         ror
         ror dy128
         dex
         bne :l1
         sta dy128+1

         lda dirx
         sta dx128
         lda dirx+1
         ldx #6
:l2      cmp #$80
         ror
         ror dx128
         dex
         bne :l2
         sta dx128+1


; println "dx128=",dx128," dy128=",dy128


  rxplus = dirx        ;rays to each side
  ryplus = diry        ;of look dir
  rxminus = dirx-DY128
  ryminus = diry+DX128

  xp = 80
  xm = 79

  ClearBitmap(buffer)

  count=80
  texture=2

  repeat

; println "dirx="dirx " diry="diry " CDX="CameraDX " CDY="CameraDY
; println "count=",count," xp=",xp," xm="xm
; println " rxp=",rxplus," ryp=",ryplus," rxm=",rxminus," rym=",ryminus
; println "  theta="theta " dirx="dirx " diry="diry

    RayCast(rxplus,ryplus,xpos,ypos)
    if RayCast_hit > 0

; println " + side=" RayCast_side " h=" RayCast_height " texturex="RayCast_tx
         ldx RayCast_hit ;map value
         lda RayCast_side
         bpl :xp
         lda ymaptex,x
         bra :psta
:xp      lda xmaptex,x
:psta    sta texture

      DrawVLine(xp, texture, RayCast_tx, &
         RayCast_height, buffer)
    endif

    RayCast(rxminus,ryminus,xpos,ypos)
    if RayCast_hit > 0

; println " - side=",RayCast_side," h=" RayCast_height " texturex="RayCast_tx

         ldx RayCast_hit ;map value
         lda RayCast_side
         bpl :xm
         lda ymaptex,x
         bra :msta
:xm      lda xmaptex,x
:msta    sta texture

      DrawVLine(xm, texture, RayCast_tx, &
         RayCast_height, buffer)
    endif

;
; Take a step in the + and - dirs
;
    rxplus = rxplus+DY128
    ryplus = ryplus-DX128
    xp++

    rxminus = rxminus-DY128
    ryminus = ryminus+DX128
    xm--

    count--
  until count=0

endsub


*-------------------------------
*
* Ray caster
*
* Input variables:
*   dx,dy - ray direction
*         - signed 16-bit
*   xpos,ypos - 16-bit position in map
*             - 8.8 fixed-point
*
* Output variables:
*   height - wall height (1/distance)
*          - 0 if no wall hit (exceeded max iterations)
*   side - high bit clr -> hit x side
*        - high bit set -> hit y side
*   hit  - zero if no hit, map val if hit
*   tx   - texture x index
*
*-------------------------------

sub RayCast(int dx, int dy, &
         int xpos, int ypos)_uint height, byte side, ubyte hit, ubyte tx

  ubyte quadrant
  byte iter           ;iteration number
  ubyte lx,ly         ;distance to first edge
  uint lxdx,lydy      ;pathlength distance to map edge
  uint invdx,invdy    ;1/dx,1/dy
  uint deltaxy        ;difference
  byte sign ;sign of deltaxy
  ubyte mapx, mapy
  uint dz             ;d/z
  int temp,temp2
  ubyte texneg ;flag to reverse in x or y

;  println "  * dx=",dx," dy=",dy," xpos="xpos," ypos="ypos

  jsr SetDxDy         ;Set up vars
  jsr CalcDeltaXY     ;Calculate deltas

;  println " * quad=",quadrant
;  println " * dx=",dx," dy=",dy," invdx=",invdx, " invdy=",invdy
;  println " * lx=",lx," ly=",ly," lxdx=",lxdx," lydy="lydy
;  println " * deltaxy="deltaxy " sign="sign

;
; Cast the ray
;
;  hit=0
;  iter=0
;  repeat
;    if deltaxy < 0
;      mapx += 1
;      deltaxy += invdx
;    else
;      mapy += 1
;      deltaxy -= invdy
;    endif
;    iter++
;  until hit > 0
;

rcast

         ldx quadrant
         beq :rpp ;++
         dex
         beq :rpm    ;+-
         dex
         beq :rmp     ;-+

         jsr raymm
         jsr GetTextureX
         lda side     ;neg -> yside
         bpl :suby
         bra :subx

:rpp     jsr raypp
         jsr GetTextureX
         lda side     ;neg -> yside
         bpl :addy ;hit x-side, so add to ypos
         bra :addx

:rpm     jsr raypm
         jsr GetTextureX
         lda side
         bpl :suby
         bra :addx

:rmp     jsr raymp
         jsr GetTextureX
         lda side
         bpl :addy
         bra :subx

;
; Right now temp is just a delta-x/y
; so we need to add or subtract it
; from the current position depending
; on the quadrant, and we only care
; about the fractional part.
;

:addx
         lda xpos
         clc
         adc temp
         bra :tax
:addy
         lda ypos
         clc
         adc temp
         bra :tax
:subx
         lda xpos
         sec
         sbc temp
         bra :tax
:suby
         lda ypos
         sec
         sbc temp

:tax     tax          ;scale to fraction of 80
         lda xtextab,x
         sta tx

;
; Finally, we need to flip the direction
; if x or y is negative
;
         lda texneg
         beq :rts
         lda #79
         sec
         sbc tx
         sta tx

; println " delta=",temp," tx=",tx," xpos=",xpos," ypos=",ypos

:rts     rts

;
; Ray casting routines by quadrant
;
; The ray is traversed from edge to
; edge until a nonzero map entry is
; found or the maximum number of
; iterations is exceeded.
;
; The x and y map postions are stored
; in .y and .x, opposite of expected,
; to enable zp map addressing.
;
; A "fencepost" correction is added
; to the distance calculation when
; moving in the positive direction
; and on a map boundary (lx=0 or ly=0)
;

 do 0
ubyte tempa,tempx,tempy

debugxy
  php
  stx tempx
  sty tempy
  println " iter=" iter " mapx=" tempy " mapy=" tempx " deltaxy="deltaxy
;  println " deltaxy=" deltaxy " invdx=" invdx " invdy=" invdy
  ldx tempx
  ldy tempy
  plp
  rts

debugA
  sta tempx
  println "a=" tempx
  lda tempx
  rts

debugaxy
  sta tempa
  stx tempx
  sty tempy
  println " .a="tempa " .x="tempx " .y="tempy
  lda tempa
  ldx tempx
  ldy tempy
  rts

debugHeightX
  sta tempa
  stx tempx
  sty tempy
  println " x.a="tempa, " lx="lx, " dx="dx," dz="dz
  lda tempa
  ldx tempx
  ldy tempy
  rts

debugHeightY
  sta tempa
  stx tempx
  sty tempy
  println " y.a="tempa, " ly="ly " dy="dy " dz="dz
  lda tempa
  ldx tempx
  ldy tempy
  rts

debugtemp
  pha
  println "temp="temp
  pla
  rts

 fin

;
; dx,dy > 0
;
raypp

; println "raypp"
; println " deltaxy=" deltaxy " invdx=" invdx " invdy=" invdy

         ldy xpos+1   ;mapx (integer portion of xpos)
         ldx ypos+1   ;mapy

         lda #00
         sta hit
         lda #16      ;max iterations
         sta iter
         rol sign     ;C set = positive (think sbc)
:loop
         ror side     ;C contains result of sbc
         bmi :pos

         iny          ;inc mapx
         lda deltaxy
         clc
         adc invdx
         sta deltaxy
         lda deltaxy+1
         adc invdx+1
         sta deltaxy+1
         jmp :cont
:pos
         inx          ;inc mapy
         lda deltaxy
         sec
         sbc invdy
         sta deltaxy
         lda deltaxy+1
         sbc invdy+1
         sta deltaxy+1
:cont
; jsr debugxy

         lda mapylo,x ;we could optimize
         sta zp       ;but this is cleaner
         lda mapyhi,x
         sta zp+1
         lda (zp),y ;map x/y
         bne :hit

         dec iter
         bne :loop    ;.A=0
         sta hit
         sta height
         sta height+1
         rts

:hit     sta hit

; Calculate the integer map distance n

; jsr debugaxy

         lda side
         bmi :yside
:xside
         tya          ;mapx
         clc
         ldy lx       ;check for lx=0
         bne :sbc     ;set or clear c
         sec
:sbc     sbc xpos+1   ;n-1, n if lx=0
         clc ;GetHeight flag
         jmp GetHeight

:yside

; Calculate m

         txa          ;mapy
         clc
         ldy ly       ;check for ly=0
         bne :sbc2    ;set or clear c
         sec
:sbc2    sbc ypos+1   ;n-1, n if lx=0
         sec          ;GetHeight flag
         jmp GetHeight

;
; dx>0, dy<0
;
raypm

; println "raypm"
; println "deltaxy=" deltaxy " invdx=" invdx " invdy=" invdy

         ldy xpos+1   ;mapx (integer portion of xpos)
         ldx ypos+1   ;mapy

         lda #00
         sta hit
         lda #16  ;max iterations
         sta iter
         rol sign     ;C set = pos (think sbc)
:loop
         ror side     ;C contains result of sbc
         bmi :pos

         iny          ;inc mapx
         lda deltaxy
         clc
         adc invdx
         sta deltaxy
         lda deltaxy+1
         adc invdx+1
         sta deltaxy+1
         jmp :cont
:pos
         dex          ;dec mapy
         lda deltaxy
         sec
         sbc invdy
         sta deltaxy
         lda deltaxy+1
         sbc invdy+1
         sta deltaxy+1
:cont

; jsr debugxy

         lda mapylo,x ;we could optimize
         sta zp       ;but this is cleaner
         lda mapyhi,x
         sta zp+1
         lda (zp),y   ;map x/y
         bne :hit

         dec iter
         bne :loop    ;.A=0
         sta hit
         sta height
         sta height+1
         rts

:hit     sta hit

; Calculate the integer map distance n

; jsr debugaxy
         lda side
         bmi :yside
:xside
         tya          ;mapx
         clc
         ldy lx       ;check for lx=0
         bne :sbc     ;set or clear c
         sec
:sbc     sbc xpos+1   ;n-1, n if lx=0
         clc ;flag
         jmp GetHeight

:yside

; Calculate m=ypos-mapy

         stx ztemp    ;mapy
         lda ypos+1
         clc
;         ldy ly       ;check for ly=0
;         bne :sbc2    ;set or clear c
;         sec
:sbc2    sbc ztemp    ;n-1
         sec      ;flag
         jmp GetHeight

;
; dx<0, dy>0
;
raymp
; println "raymp"
         ldy xpos+1   ;mapx (integer portion of xpos)
         ldx ypos+1   ;mapy

         lda #00
         sta hit
         lda #16  ;max iterations
         sta iter
         rol sign     ;C set -> positive
:loop
         ror side     ;C contains result of sbc
         bmi :pos

         dey          ;dec mapx
         lda deltaxy
         clc
         adc invdx
         sta deltaxy
         lda deltaxy+1
         adc invdx+1
         sta deltaxy+1
         jmp :cont
:pos
         inx          ;inc mapy
         lda deltaxy
         sec
         sbc invdy
         sta deltaxy
         lda deltaxy+1
         sbc invdy+1
         sta deltaxy+1
:cont

; jsr debugxy

         lda mapylo,x ;we could optimize
         sta zp       ;but this is cleaner
         lda mapyhi,x
         sta zp+1
         lda (zp),y ;map x/y
         bne :hit

         dec iter
         bne :loop ;.A=0
         sta hit
         sta height
         sta height+1
         rts

:hit     sta hit

; Calculate the integer map distance n

; jsr debugaxy
         lda side
         bmi :yside
:xside
         sty ztemp    ;mapx
         lda xpos+1
         clc
;         ldy lx       ;check for lx=0
;         bne :sbc     ;set or clear c
;         sec
:sbc     sbc ztemp    ;n-1
         clc ;flag
         jmp GetHeight

:yside

; Calculate m

         txa          ;mapy
         clc
         ldy ly       ;check for ly=0
         bne :sbc2    ;set or clear c
         sec
:sbc2    sbc ypos+1   ;n-1, n if lx=0
         sec ;flag
         jmp GetHeight

;
; dx<0, dy<0
;
raymm
; println "raymm"

         ldy xpos+1   ;mapx (integer portion of xpos)
         ldx ypos+1   ;mapy

         lda #00
         sta hit
         lda #16  ;max iterations
         sta iter
         rol sign     ;C set=pos
:loop
         ror side     ;C contains result of sbc
         bmi :pos

         dey          ;dec mapx
         lda deltaxy
         clc
         adc invdx
         sta deltaxy
         lda deltaxy+1
         adc invdx+1
         sta deltaxy+1
         jmp :cont
:pos
         dex          ;dec mapy
         lda deltaxy
         sec
         sbc invdy
         sta deltaxy
         lda deltaxy+1
         sbc invdy+1
         sta deltaxy+1
:cont

; jsr debugxy

         lda mapylo,x ;we could optimize
         sta zp       ;but this is cleaner
         lda mapyhi,x
         sta zp+1
         lda (zp),y ;map x/y
         bne :hit

         dec iter
         bne :loop    ;.A=0
         sta hit
         sta height
         sta height+1
         rts

:hit     sta hit

; Calculate the integer map distance n

; jsr debugaxy
         lda side
         bmi :yside
:xside
         sty ztemp   ;mapx
         lda xpos+1
         clc
;         ldy lx       ;check for lx=0
;         bne :sbc     ;set or clear c
;         sec
:sbc     sbc ztemp    ;n-1
         clc ;flag
         jmp GetHeight

:yside

; Calculate m

         stx ztemp    ;mapy
         lda ypos+1
         clc
;         ldy ly       ;check for ly=0
;         bne :sbc2    ;set or clear c
;         sec
:sbc2    sbc ztemp    ;n-1
         sec          ;flag
;         jmp GetHeight

;
; Compute height as scale*1/distance
; where
;     1/d = dx / (n-1 + lx/256)
; with
;   dx = set by SetDxDy
;   n = number of map steps in x or y
;   lx = set by SetDxDy
; exception: if lx=0 then 1/d = 1/n,
; i.e. we are exatly on a map boundary
;
;
; This operation is a projection,
; d*x/z, where x and z are 16-bits,
; and therefore we can use the lib3d
; projection algorithm for high
; accuracy.  (Using the log division
; routines results in low accuracy.)
;
; x*d/z = (xh + xl/256)*(N + R/256)
;       = xh*N + xh*R/256 + xl*N/256 + xl*R/256^2
;
; z and x are shifted until z is 8-bit
; then d/z is computed by lookup.  The
; result is an integer and remainder
; that then multiplies (shifted) x.
;
; A 16-bit height is calculated since
; it can be greater than 256 when
; close to a wall.
;
; On entry
;   .A contains integer distance
;   lx contains fractional distance
;   dx contains ubyte (xh=0)
;   (same for ly and dy)
;   C clear=x-side, C set=y-side
;

GetHeight
         ldx #00
         stx temp     ;height
         stx temp+1

         bcc xside
         jmp yside

;-------------------------------
;
; We hit in the x-direction
;
;-------------------------------

xside

; DEBUG
; jsr debugHeightX
; jsr debugaxy

; Scale dx and z to make z 8bit with
; maximum resolution
; .A = integer z, lx = decimal z

         tax          ;integer part
         beq :zdone

:zloop
         lsr dx+1
         ror dx
         lsr
         ror lx ;this will be Z
         tax      ;until .A=0
         bne :zloop

:zdone                ;z = n-1 + lx

; Get d/z

         ldx lx
         lda invzlo,x
         sta dz
         lda invzhi,x
         sta dz+1     ;d/z = N + R/256

; DEBUG
; jsr debugHeightX

;
; Now do the multiplies
;
; dx*d/z = xh*N + xh*R/256 + xl*N/256 + xl*R/256^2
;
         jsr initfastmult ;FIXME

;first term: xh*N

         lda dx+1     ;xh
         beq :skip1   ;if 0 then done
         ldy dz+1     ;N should be always > 0, but just in case
         beq :skip0
         jsr fastmult
         sta temp+1
         stx temp
:skip0

;second term: xh*R/256

         ldy dz       ;R
         beq :skip1
         jsr fastmult2 ;zp already set
         stx temp2  ;fractional part
         clc
         adc temp
         sta temp
         bcc :skip1
         inc temp+1
:skip1

;third term: xl*N/256

         lda dx       ;xl
         beq :skip3 ;if 0 then done
         ldy dz+1     ;N (always > 0)
         beq :skip2
         jsr fastmult
         pha
         txa
         clc
         adc temp2
         sta temp2
         pla
         adc temp
         sta temp
         bcc :skip2
         inc temp+1
:skip2

;fourth term: xl*R/65536

         ldy dz       ;R
         beq :skip3
         jsr fastmult2 ;xl unchanged
         clc          ;.A = high byte
         adc temp2
         sta temp2
         bcc :skip3
         inc temp
         bne :skip3
         inc temp+1

:skip3
         lda temp   ;we want to keep
;         cmp #160    ;height for
;         bcs :xmax   ;texture calcs,
         sta height
         lda temp+1
         sta height+1
         rts

;:xmax    lda #255     ;max height
;         sta height
;         rts

;-------------------------------
;
; We hit in the y-direction
;
;-------------------------------

yside

; DEBUG
; jsr DebugHeightY
; jsr debugaxy

; Scale dy and z to make z 8bit
; .A = integer z, lx = decimal z

         tay
         beq :zdone

:zloop
         lsr dy+1
         ror dy
         lsr
         ror ly       ;this will be Z
         tax      ;until .A=0
         bne :zloop

:zdone                ;z = n-1 + lx

; Get d/z

         ldx ly
         lda invzlo,x
         sta dz
         lda invzhi,x
         sta dz+1     ;d/z = N + R/256

; DEBUG
; jsr debugHeightY

;
; Now do the multiplies
;
; x*d/z = xh*N + xh*R/256 + xl*N/256 + xl*R/256^2
;

         jsr initfastmult ;FIXME

;first term: yh*N

         lda dy+1     ;yh
         beq :skip1   ;if 0 then done
         ldy dz+1     ;N should be always > 0, but just in case
         beq :skip0
         jsr fastmult
         sta temp+1
         stx temp
:skip0

;second term: yh*R/256

         ldy dz       ;R
         beq :skip1
         jsr fastmult2 ;zp already set
         stx temp2  ;fractional part
         clc
         adc temp
         sta temp
         bcc :skip1
         inc temp+1
:skip1

;third term: yl*N/256

         lda dy       ;yl
         beq :skip3 ;if 0 then done
         ldy dz+1     ;N (always > 0)
         beq :skip2
         jsr fastmult
         pha
         txa
         clc
         adc temp2
         sta temp2
         pla
         adc temp
         sta temp
         bcc :skip2
         inc temp+1
:skip2

;fourth term: yl*R/65536

         ldy dz       ;R
         beq :skip3
         jsr fastmult2 ;xl unchanged
         clc          ;.A = high byte
         adc temp2
         sta temp2
         bcc :skip3
         inc temp
         bne :skip3
         inc temp+1

:skip3
         lda temp   ;we want to keep
         sta height
         lda temp+1
         sta height+1
         rts

;
; Multiply .A times .Y
; Result in .A (hi) and .X (lo)
; On exit, Z, N etc flags set to .A
;

initfastmult
         pha
         lda #>multlo1
         sta zp3+1    ;f(a-b)
         lda #>multlo2
         sta zp1+1    ;f(a+b)
         lda #>multhi1
         sta zp4+1    ;f(a-b)
         lda #>multhi2
         sta zp2+1    ;f(a+b)
         pla
         rts

fastmult
         sta zp1      ;lo a+b
         sta zp2      ;hi a+b
         eor #$ff
         clc
         adc #1
         sta zp3      ;lo a-b
         sta zp4      ;hi a-b
fastmult2
         lda (zp1),y  ;f(a+b), low
         sec
         sbc (zp3),y  ;f(a-b), low
         tax
         lda (zp2),y  ;f(a+b), high
         sbc (zp4),y  ;f(a-b), high
         rts

*-------------------------------
;
; Get the x-index into the texture.
;
; The x-coordinate is given by
;    dy/dx * (n + lx)
; which is
;    dy * d/height
; which we can calculate as in
; GetHeight above, including shifting
; the result if height>255.
;
; Since we only care about the
; fractional part, we only compute
; those terms.
;
; To get the x-coord we then
; multiply the above by 80/256
;
; First, do it in Slang.  Then convert
; to asm.
;
*-------------------------------

 do 0
TestTX
         lda #123
         sta height
         lda #$80
         sta side
         lda #12
         sta dy
         lda #01
         sta dy+1

         jsr initfastmult
 fin

;
; Place dx or dy into temp2, then
; shift if height > 255.
;
; We also need to flip the texture
; when hitting +x or -y sides to
; make things look right.
;

GetTextureX

         lda side
         bmi :yside

         lda dy
         sta temp2
         lda dy+1
         sta temp2+1
         lda quadrant
         and #2       ;Do we need to
         eor #2
         sta texneg   ;flip x or y
         bra :c0

:yside   lda dx
         sta temp2
         lda dx+1
         sta temp2+1
         lda quadrant
         and #1
         sta texneg
:c0

; println " gtx: height=",height," dxy=",temp2," dz=",dz

         lda height
         sta temp
         lda height+1 ;integer part
         beq :zdone
:zloop
         lsr temp2+1
         ror temp2
         lsr
         ror temp     ;this will be Z
         tax      ;until .A=0
         bne :zloop

:zdone                ;z = n-1 + lx
         ldx temp ;scaled height
         lda invzlo,x ;d/height
         sta dz
         lda invzhi,x
         sta dz+1
         stz temp
;
; Now multiply temp2*dz
;
; x*dz = xh*N + xh*R/256 + xl*N/256 + xl*R/256^2
;
; where x=dx or dy depending on which
; side was hit. The result is an int
; plus fraction, and since we only
; care about the fraction that's all
; we compute:
;
;   lo byte xh*N + hi byte of middle two terms
;
; Note that although GetHeight does
; a shift on dx or dy, the one we
; want has not been shifted.
;

;         jsr initfastmult ;FIXME

;first term: xh*N low byte

         lda temp2+1
         beq :skip2
         ldy dz+1 ;N
         beq :skip1
         jsr fastmult
         stx temp     ;Only care about low byte
:skip1

;second term: xh*R/256

         ldy dz       ;R
         beq :skip2
         jsr fastmult2
         clc ;high byte
         adc temp
         sta temp
:skip2

;third term: xl*N/256 hi byte

         lda temp2   ;xl
         beq :skip3
         ldy dz+1    ;N
         beq :skip3
         jsr fastmult
         clc
         adc temp     ;hi byte
         sta temp
:skip3

; println " dx="dx, " dy="dy, " side="side
; println " height=",height," dz=",dz," temp2=",temp2," temp="temp

         rts



;
; Set up variables: figure out which
; quadrant we are in, and set up
; dx,dy,lx,ly, make them all positive.
;
; We actually only care about the
; high byte (integer portion) of dx
; and dy
;
; quadrants are
;   0 - dx>0, dy>0
;   1 - dx>0, dy<0
;   2 - dx<0, dy>0
;   3 - dx<0, dy<0
;
SetDxDy

  if dx<0
    quadrant = 2
    dx = -dx
         lda xpos    ;edge is to the left, dist=decimal part of xpos
         sta lx
  else
    quadrant = 0
         lda #00      ;edge is to the right, dist=256-xpos
         sec
         sbc xpos ;just the lower 8-bits
         sta lx
  endif

  if dy<0
    quadrant++
    dy = -dy
    lda ypos
    sta ly
  else
    lda #00
    sec
    sbc ypos
    sta ly
  endif

         rts


;
; Calculate path length distance to x and y edges
; lx/dx and ly/dy
;
; dx and dy are made positive in
; SetDxDy above
;
; On exit:
;   invdx, invdy are set
;   deltaxy is calculated
;   sign is stored in "sign"
;

CalcDeltaXY

; Compute 256/dx and 256/dy
;
; When dx or dy are small the lookup
; is inaccurate, so we first condition
; the numbers to be large, and then
; reverse it after the lookup.

         ldx dx
         lda dx+1
         jsr getinv
         stx invdx
         sta invdx+1

         ldx dy
         lda dy+1
         jsr getinv
         stx invdy
         sta invdy+1

; println "_ dx=",dx," dy=",dy," invdx=",invdx, " invdy=",invdy

;
; Compute lx/dx and ly/dy
;
; Use the same strategy as for
; projection:
;   lx*(N+R/256)=lx*N + lx*R/256
;
; however, lx is the distance to the
; next edge so lx=0 is actually lx=256
; and needs to be handled special. The
; rule is
;   if moving in the positive dir,
;     lx/dx = 256/dx
;   if moving in the negative dir,
;     lx/dx = 0
;

         jsr initfastmult ;FIXME

         lda ly
         beq :yzero
:ldy1    ldy invdy+1
         jsr fastmult ;ly*N
         sta lydy+1
         stx lydy
         ldy invdy
         jsr fastmult2 ;ly*R hi
         clc
         adc lydy
         sta lydy
         bcc :c1
         inc lydy+1
         bra :c1
;
; if ly=0, we need to set lydy
; based on the ray direction
;
:yzero
         lda quadrant
         and #1
         bne :negy
         lda invdy
         sta lydy
         lda invdy+1
         sta lydy+1
         bra :c1
:negy    stz lydy
         stz lydy+1
:c1
         lda lx
         beq :xzero
         ldy invdx+1
         jsr fastmult ;lx*N
         sta lxdx+1
         stx lxdx
         ldy invdx
         jsr fastmult2 ;lx*R hi
         clc
         adc lxdx
         sta lxdx
         bcc :c2
         inc lxdx+1
         bra :c2
:xzero
         lda quadrant
         and #2
         bne :negx
         lda invdx
         sta lxdx
         lda invdx+1
         sta lxdx+1
         bra :c2
:negx    stz lxdx
         stz lxdx+1

:c2


; compute deltaxy = lx/dx - ly/dy
; (16-bit result for resolution)

         lda lxdx
         sec
         sbc lydy
         sta deltaxy
         lda lxdx+1
         sbc lydy+1
         sta deltaxy+1 ;00 or $ff
         ror sign ;high bit set/clear

; DEBUG
; println "^ dx=",dx," dy=",dy," invdx=",invdx, " invdy=",invdy
; println "^ lx=",lx," ly=",ly," lxdx=",lxdx," lydy="lydy
; println "^ deltaxy="deltaxy

         rts

;
; Conditon .ax, look up the inverse,
; then condition the result.
;
; On exit,
;  .A=hi, .X=lo
;
getinv
         ldy #00      ;Number of shifts
;         cmp #16      ;We handle small and large
;         bcc :small

;         cmp #64
;         bcs :x1

         cmp #00      ;if .AX < 1/2 then
         bne :c1      ;invdx = maxint
         cpx #$80
         bcc :max
:c1
         stx zp
:l1      iny
         asl zp
         rol
         bpl :l1
:x1      tax

         lda invtablo,x
         sta zp
         lda invtabhi,x ;hi byte

         cpy #00
         beq :done
:l2      asl zp ;now fix it up
         rol
         dey
         bne :l2

         cmp #00
         bne :done ;if 17 bits
         bcs :max     ;then set to max

:done    ldx zp       ;lo byte
         rts

:max
         lda #$ff     ;deltaxy is signed
         ldx #$ff     ;17 bits
         rts



*
* Texture mapping test
*
* Textures are 160x80 four-color
* bitmaps.
*
* Textures are stored in column order
* up in banks 2+, one per bank.
*
* To render the textures an index
* table is set up in zero page, which
* is then used with a byte blaster to
* put it into the bitmap.
*
* This is test code, to get the basics
* working before integrating into the
* raycaster code.
*
* SLJ 1/26/2024
*

;         org $4000


InitGfx
  InitBlaster()
  InitScaleTables()
  rts

*-------------------------------
;
; This is all now test code below
;

 do 0
TestCode

  ubyte dev@$ba
  ubyte x,x1,textureX
  ubyte buf
  uint height

;  load "eagle80x.bitmap",dev,$8000
;  load "redbrick.bitmap",dev,$8000
;  load "purplest.bitmap",dev,$8000
;  load "colorsto.bitmap",dev,$8000
;  load "redsmile.bitmap",dev,$8000
  load "angrysmi.bitmap",dev,$8000
;  load "joker2.bitmap",dev,$8000
;  load "face80x1.bitmap",dev,$8000
;  load "marilyn8.bitmap",dev,$8000
;  load "blues80x.bitmap",dev,$8000

  AddTexture(2)

  InitBlaster()
  InitScaleTables()

; do 0
  InitBitmaps(black, red, blue, gray2)
  ClearBitmap(1)
  SetVICBuffer(1)
; fin


 buf=1
 for height=1:1000
  ClearBitmap(buf)
  for x=0:79
    textureX = x
    x1 = x+41
;    height = 512       ;height in pixels

    DrawVLine(x1,2,textureX,height,buf)
  next
  SetVICBuffer(buf)
  lda buf
  eor #$01
  sta buf
 next
  repeat
  forever

 fin
*-------------------------------


*
* Test the texture, to see if we
* unwrapped it correctly, to test
* height scaling, etc.
*
 do 0
sub TestTexture()

  ubyte x,y,pixel,height
  ubyte y1,y2,yidx
  ubyte xtex
  uint scale,ytex
  uint addr@$fa
  uint base(4)

  base(0) = $0000
  base(1) = $4000
  base(2) = $8000
  base(3) = $c000

  lda #$02
  sta texp+2
  lda #00
  sta addr

  for x=0:39
    height = 80
    scale = 256.0*159.0/(height-1)
    y1 = 80 - height/2
    y2 = y1 + height - 1
    ytex = 0
    xtex = x+x

    for y=y1:y2
      texp = xtex*160    ;bug - have to split this in two
      texp += base(x and 3)
      addr = $6000 + (y and 248)*40 + (y and 7) + (2*x and 504)

      ldy ytex+1
      lda [texp],y
      ora [addr]
      sta [addr]

      ytex += scale
    next
  next
endsub
 fin

*
* Create a texture from a bitmap
* stored at $8000
*
* Each texture is 160 rows x 80 cols
* packed into the bitmap format with
* four pixels per byte.
*
* .A = texture number (bank=.A+3)
* (this frees up banks 0-2 for user stuff)
*

sub SetTexture(@a)    ;split out to be
         clc
         adc #3
         sta texp+2   ;callable elsewhere
         rts

sub AddTexture(@a)
  ubyte x,y
  ubyte temp,pixel
  uint addr@$fa
;  uint base(4)
  ubyte mask(4)

  jsr SetTexture
  stz addr+2          ;relocatable code

;  base(0) = $0000
;  base(1) = $4000
;  base(2) = $8000
;  base(3) = $c000

  mask(0) = $c0
  mask(1) = $30
  mask(2) = $0c
  mask(3) = $03

;
; set up pointers
;
; we store four copies, each at one of
; the four possible bit offsets
;
addtex

; run through the bitmap and expand

  for x=0:79
    for y=0:159
;      texp = x*160 + y

      ldx x
      lda lo160,x
      sta texp
      lda hi160,x
      sta texp+1

;      addr = $8000 + (y and 248)*40 + (y and 7) + (x*2 and 504)

      stz addr+1
      lda y
      and #$f8
      asl
      rol addr+1
      asl
      rol addr+1
      asl
      rol addr+1      ;(y and 248)*8
      tay
      ldx addr+1

      asl
      rol addr+1
      asl
      rol addr+1
      sta addr        ;(y and 248)*32

      tya
      clc
      adc addr
      sta addr
      txa
      adc addr+1
      sta addr+1      ;(y and 248)*40

      lda y
      and #$07
      clc
      adc addr
      sta addr
      lda addr+1
      adc #$80
      sta addr+1
:c0                   ;$8000 + (y and 248)*40 + (y and 7)
      lda x
      asl             ;x*2, c clear
      and #$f8        ;x*2 and 504=$01f8
      adc addr
      sta addr
      lda addr+1
      adc #00
      sta addr+1

      temp = mask(x and 3)

      lda [addr]
      and temp
      sta pixel

 do 0
      if (x=0)
        println "x=",x," y=",y," texp="texp," addr=",addr," pixel="pixel
      endif
 fin

      lda x
      and #3
      tax
      beq :c1
:l1   asl pixel     ;place bits in 1st col
      asl pixel
      dex
      bne :l1

:c1   lda pixel
      ldy y
      jsr store
    next
  next
endsub


store
      sta temp

      sta [texp],y
      lsr
      lsr
      sta temp

;      texp += $4000
      lda texp+1
      clc
      adc #$40
      sta texp+1
      lda temp
      sta [texp],y
      lsr
      lsr
      sta temp

;      texp += $4000
      lda texp+1
      clc
      adc #$40
      sta texp+1
      lda temp
      sta [texp],y
      lsr
      lsr
      sta temp

;      texp += $4000
      lda texp+1
      clc
      adc #$40
      sta texp+1
      lda temp
      sta [texp],y

      rts

********************************
*
* Initialize scale tables
*
* Tables are stored in expansion
* ram, one per bank at $ff00-$ffff.
*
* These tables map texture y-coords
* to screen y-coords, based on the
* height.  By storing one per page
* we can set DBR to choose the
* right one.
*
* The tables are stored in order
* starting at bank 2, to leave bank 1
* open. The tables go up to around
* 240 or so, to leave the last upper
* pages alone.
*
********************************

sub InitScaleTables()

  uint height
  uint step,ytex
  ubyte y1,texh

;
; Set up scaled texture table
;

setscale
  for height=1:1023
    if height>1
      step = 256.0*159.0/(height-1) ;fencepost
    endif

    if height<161
      lda #161
      sec
      sbc height
      lsr
      sta y1            ;round(80-h/2)
    else
      stz y1
    endif

;    texp=$ff00
    stz texp
    lda height+1
    clc
    adc #$fc
    sta texp+1        ;$fc00,$fd00,...

;
; Set the starting texture index
;
; If step<1 then we need to start
; inside the texture, at 256*(80-80*step/256)
;
    ytex=0
    texh=0

    if height>160
      ytex = 20480 - 80*step
      texh = 160     ;captures height>256
    endif

; println "height=",height," step=",step," texp=",texp," thi="thi
; println "height=",height," step=",step," ytex=",ytex

    lda height
    bne :c0           ;height lo must be >0
    inc
:c0 cmp #239
    bcc :i0
    lda #238
:i0 inc               ;2..239
    sta texp+2
    cmp #2 ;treat h=1 special
    bne :c1
    lda height+1
    bne :c1
    lda #80
    sta ytex+1        ;use center point
:c1
    ldx texh
    ldy y1
    lda ytex+1
:l1 sta [texp],y
    lda ytex
    clc
    adc step
    sta ytex
    lda ytex+1
    adc step+1
    sta ytex+1 ;this is the step
    iny
    dex
    bne :l1
  next
endsub


*-------------------------------
;
; Draw a vertical line
;  x  - xcoord
;  tx - texture index 0..79
;  height - line height
;
; In multicolor mode these can all
; be bytes
;
; All rendering is done to a single
; buffer, which is later copied to
; bank 0 when done.
;
*-------------------------------

rts1     rts

sub DrawVLine(byte x@$f7, byte texnum@$f8, byte tx@$f9, &
         uint height@$fa, ubyte buffer@$fc)

  uint scale,ytex
  ubyte y1,temp
  ubyte h0,hoffset,dbr

         lda texnum
         jsr SetTexture

vbuf
;
; Set the height parameters. Height
; can now be >256 so we have to check
; for a number of special cases:
;   height = 0
;   height < 1023
;   height > 256
;   height lo byte > 238 (we can't use the higher SuperRAM banks)
;   height high byte sets the routine offset
;

         lda height ;check height=0
         ora height+1
         beq rts1

         lda height+1 ;height must be <1024
         cmp #4
         bcc :c0
         lda #$238
         sta height
         lda #3       ;max 1023
         sta height+1
:c0
         lda height
         cmp #239     ;max height lo
         bcc :c00
         lda #238
         sta height
:c00
         lda height+1 ;check > 256
         bne :lda
         lda height
         cmp #161
         bcc :sta
:lda     lda #160
:sta     sta h0       ;screen pixel height

;
; Set up texture pointer
;
;         texp = tx*160  ;column offset

         ldx tx
         cpx #80
         bcc :cx1
         ldx #79
:cx1     lda lo160,x
         sta texp
         lda hi160,x
         sta texp+1

         lda x ;add in base
         and #03
         tax
         lda :textab,x
         clc
         adc texp+1
         sta texp+1

; set by SetTexture
;         lda #$02     ;set bank
;         sta texp+2

;
; Get entry point, based on height
;
         ldx height+1
         lda :vaddr,x
         sta hoffset

         lda #161
         sec
         sbc h0
         lsr
         tay          ;round(80-h/2)

         lda buffer
         bne :buf1
:buf0
         lda x
         bmi :code2
:code1
         lda Code1RowLo,y
         sta :jmpaddr+1
         lda Code1RowHi,y
         clc
         adc hoffset ;different routines for different heights
         sta :jmpaddr+2
         bra :next
:code2
         lda Code2RowLo,y
         sta :jmpaddr+1
         lda Code2RowHi,y
         clc
         adc hoffset
         sta :jmpaddr+2
         bra :next

:buf1
         lda x
         bmi :code4
:code3
         lda Code3RowLo,y
         sta :jmpaddr+1
         lda Code3RowHi,y
         clc
         adc hoffset
         sta :jmpaddr+2
         bra :next
:code4
         lda Code4RowLo,y
         sta :jmpaddr+1
         lda Code4RowHi,y
         clc
         adc hoffset
         sta :jmpaddr+2
:next

;
; Store column offset in .X
;
         lda x
         and #$7c
         asl
         tax
;
; Draw the line
;
         sei          ;nonzero dbr
         lda $01
         pha
         and #$fe     ;RAM at $a000
         sta $01 ;so ORA works
         phb
         lda height
         bne :inc     ;things like height=512 have low byte=0
         inc          ;so we use texptr for 513 instead
:inc     inc
         pha
         plb
:jmpaddr jsl $020000
         plb
         pla
         sta $01
         cli
:rts     rts


:vaddr   dfb $00,$10,$20,$30 ;routine offsets vs height: $8000,$9000,etc.
:textab  dfb 0,$40,$80,$c0 ;texture offsets for bitmap columns


********************************
*
* Byte blaster with texture
*
* The texture is scaled by height,
* with locations stored in SuperRAM
* and DBR changed to point to the
* right bank.
*
* Because the height may be greater
* than 256, we have to create separate
* routines corresponding to the high
* byte of the height.
*
* Because DBR changes we need to sei
* before calling the routine.
*
********************************

;buf0 = $6000


codesize = 13         ;size of below code
bcode
         ldy textab   ;change to ldy zp
         lda [texp],y
         ora $123456,x
         sta $123456,x

sub InitBlaster()
  ubyte y1,y
  uint base,baddr
  uint zp@$f7,height@$fa,lo@$fb,hi@$fd
  uint offset

  lda #00
:l0
  sta height          ;high byte
  clc
  adc #$fc
  sta bcode+2
  jsr AddBlasterCode
  lda height
  clc
  adc #1
  cmp #4
  bne :l0
  rts

AddBlasterCode
  lda #00
  sta bcode+8
  sta bcode+12

  lda #2   ;store code in bank 2
  sta zp+2            ;bank 1 preferred but slang uses it, maybe later
  lda height
  asl
  asl
  asl
  asl
  sta offset+1        ;$1000,$2000,...
  stz offset

  base = $6000
  zp = Code1+offset
  lo = Code1RowLo
  hi = Code1RowHi
  jsr codeloop

  base = $6100
  zp = Code2+offset
  lo = Code2RowLo
  hi = Code2RowHi
  jsr codeloop

  base = $a000
  zp = Code3+offset
  lo = Code3RowLo
  hi = Code3RowHi
  jsr codeloop

  base = $a100
  zp = Code4+offset
  lo = Code4RowLo
  hi = Code4RowHi

codeloop
  for y1=0:79
    y = y1
    jsr emit
    y = 159-y1
    jsr emit
  next
  lda #$6b            ;rtl
  sta [zp]
  rts

emit
    ldy y
    sty bcode+1

    lda height        ;only store once
    bne :skip
    lda zp
    sta (lo),y
    lda zp+1
    sta (hi),y
:skip

    baddr = base + (y and 248)*40.0 + (y and 7)
    lda baddr
    sta bcode+6
    sta bcode+10
    lda baddr+1
    sta bcode+7
    sta bcode+11
    ldy #00
:l1
    lda bcode,y
    sta [zp],y
    iny
    cpy #codesize
    bne :l1
    lda zp
    clc
    adc #codesize
    sta zp
    lda #00
    adc zp+1
    sta zp+1
    rts


********************************
*
* Bitmap routines
*
********************************

buffer0 EQU $6000
buffer1 EQU $a000


;
; Buffers will be at $6000 and $A000
; with color map at $1c00 offset
;

sub InitBitmaps( &
      byte color0@$f9, byte color1@$fa, &
      byte color2@$fb, byte color3@$fc &
      )

         lda #$38
         sta $d018 ;set colormap to $1c00
         lda #%00111011
         sta $d011    ;enable bitmap mode
         lda #%00011000
         sta $d016    ;40 col multicolor

         lda color0   ;black bg
         sta $d020
         sta $d021
         lda color1    ;colormap fill
         asl
         asl
         asl
         asl
         ora color2
         ldx #00
l1
         sta $4c00,x  ;color map lo
         sta $4d00,x
         sta $4e00,x
         sta $4f00,x

         sta $8c00,x  ;color map hi
         sta $8d00,x
         sta $8e00,x
         sta $8f00,x
         inx
         bne l1

         ldx #00      ;clear color ram
:loop2   lda color3
         sta $d800,x
         sta $d900,x
         sta $da00,x
         sta $db00,x
         lda #00 ;and clear out
         sta $7900,x  ;unused bitmap
         sta $7a00,x
         sta $7b00,x
         sta $7c00,x
         sta $7d00,x
         sta $7e00,x
         sta $7f00,x
         sta $b900,x
         sta $ba00,x
         sta $bb00,x
         sta $bc00,x
         sta $bd00,x
         sta $be00,x
         sta $bf00,x
         inx
         bne :loop2
endsub

;
; Set VIC memory
;
sub SetVICBuffer(@a)  ;buffer in .a
         tax
         lda $dd00
         and #$fc
         cpx #00
         bne buf1
buf0     ora #$02     ;Buf0 in bank 1
         sta $dd00
         rts
buf1     ora #$01     ;Buf1 in bank 2
         sta $dd00
         rts
;
; fast bitmap clear
;
sub ClearBitmap(@a) ;buffer in .A

         tax
         bne fillbuf2
fillbuf1
         ldx #00
         txa
:loop    sta $6000,x
         sta $6100,x
         sta $6200,x
         sta $6300,x
         sta $6400,x
         sta $6500,x
         sta $6600,x
         sta $6700,x
         sta $6800,x
         sta $6900,x
         sta $6a00,x
         sta $6b00,x
         sta $6c00,x
         sta $6d00,x
         sta $6e00,x
         sta $6f00,x
         sta $7000,x
         sta $7100,x
         sta $7200,x
         sta $7300,x
         sta $7400,x
         sta $7500,x
         sta $7600,x
         sta $7700,x
         sta $7800,x
         inx
         bne :loop
         rts
fillbuf2
         ldx #00
         txa
:loop    sta $a000,x
         sta $a100,x
         sta $a200,x
         sta $a300,x
         sta $a400,x
         sta $a500,x
         sta $a600,x
         sta $a700,x
         sta $a800,x
         sta $a900,x
         sta $aa00,x
         sta $ab00,x
         sta $ac00,x
         sta $ad00,x
         sta $ae00,x
         sta $af00,x
         sta $b000,x
         sta $b100,x
         sta $b200,x
         sta $b300,x
         sta $b400,x
         sta $b500,x
         sta $b600,x
         sta $b700,x
         sta $b800,x
         inx
         bne :loop
         rts

;         put 'putcore.i.s'

*-------------------------------------
* 16 bit multiply and divide routines.
* Three 16 bit (two-byte) locations
* ACC, AUX and EXT must be set up,
* preferably on zero page.
*-------------------------------------

* MULTIPLY ROUTINE

* ACC*AUX -> [ACC,EXT] (low,hi) 32 bit result

]FAQ2     =   $69
]ACC      =   ]FAQ2   ;NOTE: hardcoded in sub asm core]ASCTONUM below
]AUX      =   ]ACC+2 ;to prevent phase errors
]EXT      =   ]AUX+2  ;update below if changing location

core]]MULT16
 LDA #0
         STA ]EXT+1
 LDY #$11
 CLC
:LOOP    ROR ]EXT+1
 ROR
         ROR ]ACC+1
         ROR ]ACC
 BCC :MUL2
 CLC
         ADC ]AUX
 PHA
         LDA ]AUX+1
         ADC ]EXT+1
         STA ]EXT+1
 PLA
:MUL2 DEY
 BNE :LOOP
         STA ]EXT
         RTS

sub asm core]mult16@core]]MULT16( &
    uint acc@$69, uint aux@$6b)_ &
    uint lo@$69, uint hi@$6d


