;***********************************************************************************;
;***********************************************************************************;
;
; An implementation of PET BASIC4 for the VIC-20
; By Simon Rowe <srowe@mose.org.uk>.
;
; Derived from an implementation found on www.zimmers.net based on the V-Link ROM
; written by David Foster of Richvale Telecommunications

;***********************************************************************************;
;***********************************************************************************;
;
; BASIC zero page

VERCHK	= $0A			; load/verify flag
COUNT	= $0B			; temporary byte, line crunch/array access/logic operators
GARBFL	= $0F			; garbage collected/open quote/DATA flag
LINNUM	= $14			; temporary integer
TEMPST	= $19			; descriptor stack
INDEX	= $22			; misc temp byte
FRESPC	= $35			; string utility ptr
VARNAM	= $45			; current variable name
VARPNT	= $47			; current variable address
FORPNT	= $49			; FOR/NEXT variable pointer
FAC1	= $61			; FAC1
FBUFPT	= $71			; temp BASIC execute/array pointer
CHRGET	= $73			; increment and scan memory, BASIC byte get
CHRGOT	= $79			; scan memory, BASIC byte get


;***********************************************************************************;
;***********************************************************************************;
;
; KERNAL zero page

STATUS	= $90			; I/O status byte
VERCK	= $93			; load/verify flag
XSAV	= $97			; register save
LDTND	= $98			; open file count
DFLTN	= $99			; input device number
DFLTO	= $9A			; output device number
PTR1	= $9E
TIME	= $A0			; jiffy clock
SAL	= $AC			; start pointer
EAL	= $AE			; end pointer
CMP0	= $B0
FNLEN	= $B7			; file name length
LA	= $B8			; logical file
SA	= $B9			; secondary address
FA	= $BA			; current device number
FNADR	= $BB			; file name pointer
STAL	= $C1
MEMUSS	= $C3
PNT	= $D1			; current screen line pointer
PNTR	= $D3			; cursor column
USER	= $F3			; colour RAM pointer low byte

BUF	= $0200			; input buffer
LAT	= $0259			; logical file table
FAT	= $0263			; device number table
SAT	= $026D			; secondary address table
KEYD	= $0277			; keyboard buffer
COLOR	= $0286			; current colour code
SHFLAG	= $028D			; keyboard shift/control flag
M51CDR	= $0294			; pseudo 6551 command register
IERROR	= $0300			; BASIC vector start
CBINV	= $0316			; BRK vector


;***********************************************************************************;
;
; hardware equates

	; Standard VIA
VIA1PA1		= $9111		; VIA 1 DRA


	; BASIC entrypoints
WARMST	= $C002			; BASIC warm start entry point
ERROR	= $C437			; do error #.X then warm start (via vector)
ERROR2	= $C43A			; do error #.X then warm start
MAIN2	= $C483			; BASIC warm start
LNKPRG	= $C533			; rebuild BASIC line chaining
CRNCH2	= $C57C			; crunch BASIC tokens (via vector)
NEWSTT	= $C7AE			; interpreter inner loop
GOTO	= $C8A0			; perform GOTO
REM	= $C93B			; perform REM
FRMEVL	= $CD9E			; evaluate expression
FEVAL	= $CE86			; get arithmetic element
RPACHK	= $CEF7			; scan for ")"
LPACHK	= $CEFA			; scan for "("
COMCHK	= $CEFD			; scan for ","
SYNCHR	= $CEFF			; scan for CHR$(.A)
FNDVAR	= $D0E7			; either find or create variable
ALCSPAC	= $D4F4			; make space in string memory for string
DELST	= $D6A3			; evaluate string
MAKADR	= $D7F7			; convert FAC1 to integer in temporary integer
INITBA	= $E3A4			; initialize BASIC RAM locations
FREMSG	= $E404			; print start up message
PATCH1	= $E4BC			; get secondary address and print "SEARCHING..."
PATCH2	= $E4C1			; set LOAD address if secondary address = 0
	
	; KERNAL entrypoints
INITSK	= $E518			; initialize hardware
SCRNOUT	= $E742			; output character
PUTSCRN	= $EAAA			; save character and colour to screen @ cursor
COLORSYN = $EAB2		; calculate pointer to colour RAM
IRQ	= $EABF			; IRQ handler
RSOUTSAV = $F0ED		; send byte to RS-232 buffer
RSNXTIN	= $F14F			; get byte from RS-232 buffer
FNDFLNO	= $F3CF			; find file
SETFLCH	= $F3DF			; set file details from table,.X
FCLALL	= $F3EF			; close all channels and files
SERNAME	= $F495			; send secondary address and filename
FUDTIM	= $F734			; increment real time clock
FSTOP	= $F770			; scan stop key
FE_NTOPN = $F784		; display file not open
FRESTOR	= $FD52			; restore default I/O vectors
INITMEM	= $FD8D			; initialize and test RAM
INITVIA	= $FDF9			; initialize I/O registers
NMI2	= $FEAD			; NMI handler
BREAK	= $FED2			; BRK handler
SCNKEY	= $FF9F			; scan the keyboard
STOP	= $FFE1			; scan stop key (via vector)


;***********************************************************************************;
;
; BASIC keyword token values

TK_GOTO		= $89			; GOTO token
TK_IF		= $8B			; IF token
TK_ON		= $91			; ON token
TK_TO		= $A4			; TO token
TK_THEN		= $A7			; THEN token

TK_PI		= $FF			; PI token


;***********************************************************************************;
;
; error messages

ER_STOP		= $00
ER_FNOTOPEN	= $03
ER_DEVNOTP	= $05
ER_ILLQUAN	= $0E
ER_STR2LONG	= $17


;***********************************************************************************;
;
; floating point accumulator offsets

FAC_EXPT	= $00
FAC_MANT	= $01
FAC_SIGN	= $05


;***********************************************************************************;
;
; private variables

SAVEREG		= $02			; MONITOR register save area
DSFA		= $BF			; disk status device number, $FF - no device, $00 - variables valid
PARSEST		= $C1			; disk command parameter parse state
						; bit	flag
						; ---	----
						;  7	replace file
						;  6	record length/write
						;  5	secondary drive
						;  4	primary drive
						;  3	device number
						;  2	logical file
						;  1	secondary file name
						;  0	primary file name
DRIVENO2	= $C2			; secondary disk number
DRIVENO1	= $C3			; primary disk number
RECLEN		= $033D			; record length/disk identifier flag
DISKID		= $033F			; disk identifier (2 bytes)
TEMPLEN		= $0341			; temporary length
TEMPSTR		= $0342			; temporary string
CMDBUF		= $0353			; disk command buffer


;***********************************************************************************;
;***********************************************************************************;
;
; BASIC4 ROM start

	* = $A000

	.WORD B4COLD
	.WORD B4WARM
	.BYTE "A0",'C'+$80,'B'+$80,'M'+$80


;***********************************************************************************;
;
; BASIC4 warm start entry point

B4COLD
	JSR INITMEM		; initialize and test RAM
	JSR FRESTOR		; restore default I/O vectors
	JSR B4VECTS		; initialize BASIC and KERNAL vectors
	STY DSFA		; set disk status, no device
	LDA #$00
	STA BUSFLGS		; clear bus flags
	JSR INITVIA		; initialize I/O registers
	JSR INITSK		; initialize hardware
	CLI			; enable interrupts
	JSR INITBA		; initialize BASIC RAM locations
	JSR FREMSG		; print start up message and initialize memory pointers
	LDY #$08		; set base address and index to 16 / 2
	STY PNT			; set current screen line pointer low byte
	LDA #'4'		; screen code for "4"
	STA (PNT),Y		; update banner to read "BASIC V4"

	JSR IEVIATEST		; test for IEEE interface
	LDA #$00
	BCC GOBASIC		; no hardware, default to serial bus

	JSR INITIEVIA		; initialize cartridge I/O registers
	JSR SCNKEY		; scan keyboard
	LDA SHFLAG		; get keyboard shift/control/C= flag
	AND #$02		; mask [C=] bit
	BEQ USEIE		; branch if not [C=]
	LDA #$00		; default to serial bus
	.BYTE $2C		; makes next line BIT $40A9
USEIE	LDA #$40		; default to IEEE bus

GOBASIC	JSR SETBUS		; set bus mapping
	JMP $E381		; continue normal BASIC coldstart


;***********************************************************************************;
;
; initialize BASIC and KERNAL vectors

B4VECTS
	LDY #$33		; set byte count
LA051	LDA ALLVECTS,Y		; get byte from table
	STA IERROR,Y		; save byte to RAM
	DEY			; decrement index
	BPL LA051		; loop if more to do
	RTS


;***********************************************************************************;
;
; BASIC vectors, these are copied to RAM from $0300 onwards

ALLVECTS
	.WORD ERROR2		; BASIC vector - print error message
	.WORD MAIN2		; BASIC vector - main command processor
	.WORD B4CRNCH		; BASIC vector - tokenise keywords
	.WORD B4QPLOP		; BASIC vector - list program
	.WORD B4GONE		; BASIC vector - execute next command
	.WORD FEVAL		; BASIC vector - get value from line
	
DFLTNAME
	.BYTE "0:*",$00		; default load file name
RUNCR	.BYTE "RUN",$0D		; second part of RUNTB


;***********************************************************************************;
;
; KERNAL vectors

	.WORD IRQ		; IRQ vector
#ifdef EXTRA_CMDS
	.WORD B4BREAK		; BRK vector
#else
	.WORD BREAK		; BRK vector
#endif ; EXTRA_CMDS
	.WORD NMI2		; NMI vector

	.WORD XOPEN		; KERNAL vector - open a logical file
	.WORD XCLOSE		; KERNAL vector - close a specified logical file
	.WORD XCHKIN		; KERNAL vector - open channel for input
	.WORD XCHKOUT		; KERNAL vector - open channel for output
	.WORD XCLRCHN		; KERNAL vector - close input and output channels
	.WORD XCHRIN		; KERNAL vector - input character from channel
	.WORD XCHROUT		; KERNAL vector - output character to channel
	.WORD FSTOP		; KERNAL vector - scan stop key
	.WORD XGETIN		; KERNAL vector - get character from keyboard queue
	.WORD FCLALL		; KERNAL vector - close all channels and files
	.WORD BREAK		; User vector

	.WORD XLOAD2		; KERNAL vector - load
	.WORD XSAVE2		; KERNAL vector - save


;***********************************************************************************;
;
; BASIC4 warm start entry point

B4WARM
	BIT VIA1PA1		; test VIA 1 DRA
	JSR FUDTIM		; increment the real time clock
	JSR STOP		; scan stop key
	BNE LA0A0		; if no stop then exit
	JSR B4VECTS		; initialize BASIC and KERNAL vectors
	JMP $FED5		; jump to normal warm start

LA0A0	JMP $FEFF		; restore registers and exit interrupt


;***********************************************************************************;
;
; get file details

GETLFN
	LDA #$00		; clear .A
	STA STATUS		; clear I/O status byte
	JSR FNDFLNO		; find file
	BEQ LA0B1		; branch if file opened
	PLA			; else dump calling address low byte
	PLA			; dump calling address high byte
	JMP FE_NTOPN		; do file not open error and return

LA0B1	JSR SETFLCH		; set file details from table,.X
LA0B4	LDA FA			; get device number
	CMP #$04		; compare device with first serial device
LA0B8	RTS

LA1D0	LDA #$0D		; set [CR]
LA1D2	PHA			; save character
	JSR RESTOUT		; restore previous output device
	PLA			; restore character
	JSR XCHROUT		; output character using vectored routine


;***********************************************************************************;
;
; restore previous output device

RESTOUT
	LDA EAL			; get saved output device
	CMP #$03		; compare with screen
	BEQ LA0B8		; if screen then exit
	LDX EAL+1		; get saved logical file then ..
	JMP XCHKOUT


;***********************************************************************************;
;
; crunch BASIC4 tokens, the crunch BASIC tokens vector is initialized to point here

B4CRNCH
	LDX CHRGOT+1		; get BASIC execute pointer low byte
	LDY CHRGOT+1		; get BASIC execute pointer low byte
LA4EE	JSR COPYBYTE		; copy byte from BUF,.X to BUF,.Y
	BEQ LA558		; branch if was null [EOL]
	CMP #$22		; compare with quote character
	BNE LA4FC
	JSR LA57D		; copy quoted string
	BEQ LA53F
LA4FC	CMP #'A'		; compare with "A"
	BCC LA53F		; branch if <
	CMP #'Z'+1		; compare with character after "Z"
	BCS LA53F		; branch if >=
	STY FBUFPT		; copy save index
	STX XSAV
	LDY #$01
	STY COUNT
	DEX
	LDY #$FF
LA50F	INX			; next table byte
	INY			; next buffer byte
LA511	LDA BUF,X		; get byte from input buffer
	SEC
	SBC B4WORDS,Y		; subtract table byte
	BEQ LA50F		; go compare next if match
	ASL			; shift b7 into Cb, 0 into b0
	BNE LA543		; non-zero so not end marker match, go try the next keyword
	LDY FBUFPT		; restore save index
	LDA COUNT
	CMP #VECTLSB-VECTMSB	; compare with the number of BASIC4 tokens
	BNE LA53C
	LDX XSAV
	.BYTE $2C		; makes next line BIT $C8E8
LA528	INX			; increment source index
	INY			; increment destination index
	JSR COPYBYTE		; copy byte from BUF,.X to BUF,.Y
	BEQ LA558		; branch if was null [EOL]
	CMP #':'		; compare with ":"
	BEQ LA53F
	CMP #$22		; compare with quote character
	BNE LA528
	JSR LA57D
	BEQ LA528
LA53C	STA BUF,Y
LA53F	INX
	INY
	BNE LA4EE
LA543	LDX XSAV
	INC COUNT		; increment word index (next word)

				; now find end of this word in the table

LA547	INY			; increment table index
	LDA B4WORDS-1,Y		; get table byte
	BPL LA547		; loop if not end of word yet
	LDA B4WORDS,Y		; get byte from keyword table
	BNE LA511		; go test next word if not zero byte, end of table
	LDY FBUFPT		; restore save index
	BPL LA53F
LA556	PLA
	PLA
LA558	JSR CRNCH2		; crunch BASIC2 tokens
	LDX #$00		; clear index
LA55D	LDA BUF,X
	BEQ LA57C		; if EOL then exit
	INX			; increment index
	CMP #$22		; compare with quote character
	BNE LA571
LA567	LDA BUF,X
	BEQ LA57C		; if EOL then exit
	INX			; increment index
	CMP #$22		; compare with quote character
	BNE LA567		; loop until close quote
LA571	CMP #' '
	BCS LA55D
	ADC #$CB		; add token offset
	STA BUF-1,X
	BNE LA55D
LA57C	RTS

LA57D	INX			; increment source index
	INY			; increment destination index
	JSR COPYBYTE		; copy byte from BUF,.X to BUF,.Y
	BEQ LA556		; branch if was null [EOL]
	CMP #$22		; compare with quote character
	BNE LA57D		; loop if not quote
	RTS


;***********************************************************************************;
;
; copy byte from BUF,.X to BUF,.Y

COPYBYTE
	LDA BUF,X		; get byte
	STA BUF,Y		; set byte
	RTS


;***********************************************************************************;
;
; uncrunch BASIC4 tokens, the uncrunch BASIC tokens vector is initialized to point here

B4QPLOP
	BPL LA5C1		; just go print it if not token byte
	CMP #TK_PI		; compare with the token for PI
	BEQ LA5C1		; just print it if so
	BIT GARBFL		; test the open quote flag
	BMI LA5C1		; just go print character if open quote set
	CMP #$CC		; compare with first BASIC4 token
	BCC LA5C4
	SBC #$CB		; reduce token range to 1 to whatever
	TAX			; copy token # to .X
	STY FORPNT		; save index for line
	LDY #$FF		; start from -1, adjust for pre increment
LA5A8	DEX			; decrement token #
	BEQ LA5B3		; if now found go do printing
LA5AB	INY			; else increment index
	LDA B4WORDS,Y		; get byte from keyword table
	BPL LA5AB		; loop until keyword end marker
	BMI LA5A8		; go test if this is required keyword, branch always
	
				; found keyword, it's the next one
	
LA5B3	INY			; increment keyword table index
	LDA B4WORDS,Y		; get byte from table
	BMI LA5BE		; go restore index, mask byte and print if
				; byte was end marker

	JSR $CB47		; else go print the character
	BNE LA5B3		; go get next character, branch always
LA5BE	JMP $C6EF		; restore index, mask byte and print

LA5C1	JMP $C6F3		; print non-token
LA5C4	JMP $C724		; jump to BASIC2 uncrunch


;***********************************************************************************;
;
; BASIC4 keywords. Each word has b7 set in its last character as an end marker.

B4WORDS
	.BYTE "CONCA",'T'+$80
	.BYTE "DOPE",'N'+$80
	.BYTE "DCLOS",'E'+$80
	.BYTE "RECOR",'D'+$80
	.BYTE "HEADE",'R'+$80
	.BYTE "COLLEC",'T'+$80
	.BYTE "BACKU",'P'+$80
	.BYTE "COP",'Y'+$80
	.BYTE "APPEN",'D'+$80
	.BYTE "DSAV",'E'+$80
	.BYTE "DLOA",'D'+$80
	.BYTE "CATALO",'G'+$80
	.BYTE "RENAM",'E'+$80
	.BYTE "SCRATC",'H'+$80
	.BYTE "DIRECTOR",'Y'+$80
	.BYTE "IEE",'E'+$80
	.BYTE "SERIA",'L'+$80
	.BYTE "C",'D'+$80
#ifdef EXTRA_CMDS
	.BYTE "MONITO",'R'+$80
	.BYTE "MODE",'M'+$80
	.BYTE "DAT",'A'+$80
#endif ; EXTRA_CMDS
MONITOR	.BYTE $00		; end marker


;***********************************************************************************;
;
; Action addresses for BASIC4 commands. These are called by pushing the address
; onto the stack and doing an RTS so the actual address - 1 needs to be pushed.

VECTMSB	.BYTE >CONCAT-1		; perform CONCAT (MSB)
	.BYTE >DOPEN-1		; perform DOPEN (MSB)
	.BYTE >DCLOSE-1		; perform DCLOSE (MSB)
	.BYTE >RECORD-1		; perform RECORD (MSB)
	.BYTE >HEADER-1		; perform HEADER (MSB)
	.BYTE >COLLECT-1	; perform COLLECT (MSB)
	.BYTE >BACKUP-1		; perform BACKUP (MSB)
	.BYTE >COPY-1		; perform COPY (MSB)
	.BYTE >APPEND-1		; perform APPEND (MSB)
	.BYTE >DSAVE-1		; perform DSAVE (MSB)
	.BYTE >DLOAD-1		; perform DLOAD (MSB)
	.BYTE >CATALOG-1	; perform CATALOG (MSB)
	.BYTE >RENAME-1		; perform RENAME (MSB)
	.BYTE >SCRATCH-1	; perform SCRATCH (MSB)
	.BYTE >CATALOG-1	; perform DIRECTORY (MSB)
	.BYTE >IEEE-1		; perform IEEE (MSB)
	.BYTE >SERIAL-1		; perform SERIAL (MSB)
	.BYTE >CD-1		; perform CD (MSB)
#ifdef EXTRA_CMDS
	.BYTE >MONITOR-1	; perform MONITOR (MSB)
	.BYTE >MODEM-1		; perform MODEM (MSB)
	.BYTE >DATA-1		; perform DATA (MSB)
#endif ; EXTRA_CMDS

VECTLSB	.BYTE <CONCAT-1		; perform CONCAT (LSB)
	.BYTE <DOPEN-1		; perform DOPEN (LSB)
	.BYTE <DCLOSE-1		; perform DCLOSE (LSB)
	.BYTE <RECORD-1		; perform RECORD (LSB)
	.BYTE <HEADER-1		; perform HEADER (LSB)
	.BYTE <COLLECT-1	; perform COLLECT (LSB)
	.BYTE <BACKUP-1		; perform BACKUP (LSB)
	.BYTE <COPY-1		; perform COPY (LSB)
	.BYTE <APPEND-1		; perform APPEND (LSB)
	.BYTE <DSAVE-1		; perform DSAVE (LSB)
	.BYTE <DLOAD-1		; perform DLOAD (LSB)
	.BYTE <CATALOG-1	; perform CATALOG (LSB)
	.BYTE <RENAME-1		; perform RENAME (LSB)
	.BYTE <SCRATCH-1	; perform SCRATCH (LSB)
	.BYTE <CATALOG-1	; perform DIRECTORY (LSB)
	.BYTE <IEEE-1		; perform IEEE (LSB)
	.BYTE <SERIAL-1		; perform SERIAL (LSB)
	.BYTE <CD-1		; perform CD (LSB)
#ifdef EXTRA_CMDS
	.BYTE <MONITOR-1	; perform MONITOR (LSB)
	.BYTE <MODEM-1		; perform MODEM (LSB)
	.BYTE <DATA-1		; perform DATA (LSB)
#endif ; EXTRA_CMDS


;***********************************************************************************;
;
; Disk command templates. Describes the command letter, punctuation and parameter
; order for each DOS command.

CMDTEMPL
	.BYTE '$',$D1,':',$F1
	.BYTE $D1,':',$F1,',',$E1,',',$E0
	.BYTE $D1,':',$F1,',','A'
	.BYTE 'N',$D1,':',$F1,',',$D0
	.BYTE 'V',$D1
	.BYTE 'D',$D2,'=',$D1
	.BYTE 'C',$D2,':',$F2,'=',$D1,':',$F1
	.BYTE 'C',$D2,':',$F2,'=',$D2,':',$F2,',',$D1,':',$F1
	.BYTE 'R',$D1,':',$F2,'=',$D1,':',$F1
	.BYTE 'S',$D1,':',$F1
	.BYTE 'C','D',':',$F1


;***********************************************************************************;
;
; start new BASIC code, the start new BASIC code vector is initialized to point here

B4GONE
	LDA CHRGOT+1		; get BASIC execute pointer low byte
	PHA			; save it
	LDA CHRGOT+2		; get BASIC execute pointer high byte
	PHA			; save it
LA6B1	JSR CHRGET		; increment and scan memory
	BEQ LA6D2		; branch if no more chrs
	CMP #$22		; compare with quote character
	BNE LA6C3		; branch if not quote
LA6BA	JSR CHRGET		; increment and scan memory
	BEQ LA6D2		; branch if no more chrs
	CMP #$22		; compare with quote character
	BNE LA6BA		; loop until end quote
LA6C3	CMP #'D'		; compare character with "D"
	BNE LA6B1
	LDY #$01
	LDA (CHRGOT+1),Y	; get next BASIC byte
	CMP #'S'		; compare next character with "S"
	BNE LA6B1		; branch if not "DS" variable
	LDA DSFA		; get disk status device number
LA6D2	BEQ LA711		; if disk status valid continue parsing
	BMI LA6E9		; if no device use dummy values
	LDX #$00		; clear .A
	STX STATUS		; clear I/O status byte
	JSR XTALK		; command a device on either the serial or IEEE bus to TALK
	LDA STATUS		; get I/O status byte
	BEQ LA6F9		; if no error read disk status
LA6E9	LDX #'1'
	STX CMDBUF		; store '1'
	DEX
	STX CMDBUF+1		; store '0'
	LDX #$00
	STX CMDBUF+2		; store NUL
	BEQ LA713		; set value of DS$, branch always


;***********************************************************************************;
;
; read disk status from command channel

LA6F9	LDA #$6F		; set secondary address to 15 (command channel)
	JSR XTKSA		; send secondary address after TALK
	LDX #$00		; clear index
LA700	JSR XACPTR		; input a byte from either the serial or IEEE bus
	STA CMDBUF,X		; save byte in buffer
	INX			; increment index
	LDA STATUS		; get I/O status byte
	BEQ LA700		; loop while no error
	JSR XUNTLK		; command the serial or IEEE bus to UNTALK
	DEX			; decrement buffer length
	BEQ LA6E9		; if empty buffer use dummy values

LA711	BEQ LA762		; branch always


;***********************************************************************************;
;
; store disk status in DS$

LA713	TXA			; copy string length
	PHA			; save string length
	LDA #'D'		; set first character to "D"
	STA VARNAM		; set variable name 1st character
	LDA #'S'+$80		; set second character to "S$"
	STA VARNAM+1		; set variable name 2nd character
	JSR FNDVAR		; find or create variable
	PLA			; restore string length
	PHA			; save string length
	JSR ALCSPAC		; make space in string memory for string
	LDY #$02
	LDA FRESPC+1		; get string utility ptr high byte
	STA (VARPNT),Y		; set string location high byte
	LDA #>CMDBUF		; get status message high byte
	STA CHRGOT+2		; save BASIC execute pointer high byte
	DEY			; decrement index
	LDA FRESPC		; get save string utility ptr low byte
	STA (VARPNT),Y		; set string location low byte
	LDA #<CMDBUF		; get status message low byte
	STA CHRGOT+1		; save BASIC execute pointer low byte
	DEY			; decrement index
	STY DSFA		; set disk status variables valid
	PLA			; restore string length
	STA (VARPNT),Y		; save string length
	TAY			; copy string length to .Y
LA740	DEY			; decrement index
	BMI LA74A		; if empty string store error code
	LDA CMDBUF,Y		; get character
	STA (FRESPC),Y		; save character
	BNE LA740		; loop while characters left


;***********************************************************************************;
;
; store disk error code in DS

LA74A	LDA #'S'		; set second character to "S"
	STA VARNAM+1		; set variable name 2nd character
	JSR FNDVAR		; find or create variable
	JSR FRMEVL		; evaluate expression
	ASL FAC1+FAC_MANT	; shift FAC1 mantissa
	LSR FAC1+FAC_MANT	; clear FAC1 sign bit
	LDY #$04		; set index
LA75A	LDA !FAC1,Y		; get FAC1,.Y
	STA (VARPNT),Y		; save byte to variable
	DEY			; decrement index
	BPL LA75A		; loop until value copied
LA762	PLA			; restore previous BASIC execute pointer high byte
	STA CHRGOT+2		; save BASIC execute pointer high byte
	PLA			; restore previous BASIC execute pointer low byte
	STA CHRGOT+1		; save BASIC execute pointer low byte
	JSR CHRGET		; increment and scan memory
	.BYTE $2C		; makes next line BIT $6868
LA76C	PLA
	PLA
	JSR CHRGOT		; scan memory
	BEQ LA788		; branch if no more chrs
	CMP #TK_IF		; compare with token for IF
	BNE LA779
	LDA #$E0
LA779	SEC
	SBC #$CC		; subtract first BASIC4 token
	BCC LA788		; branch if less than the first token
	CMP #VECTLSB-VECTMSB	; compare with the number of BASIC4 tokens
	BCS LA788		; branch if greater
	JSR LA78E		; go interpret token
	JMP NEWSTT		; go do interpreter inner loop

LA788	JSR CHRGOT		; scan memory
	JMP $C7E7		; go interpret BASIC code and loop

LA78E	TAY			; copy to index
	LDA VECTMSB,Y		; get vector high byte
	PHA			; push on stack
	LDA VECTLSB,Y		; get vector low byte
	PHA			; push on stack
	JMP CHRGET		; increment and scan memory and return. the return in
				; this case calls the command code, the return from
				; that will eventually return to the interpreter inner
				; loop


#ifdef EXTRA_CMDS
;***********************************************************************************;
;
; perform DATA

DATA
	JSR FRMEVL		; evaluate expression
	JSR CHRGOT		; scan memory
	CMP #TK_GOTO		; compare with token for GOTO
	BEQ LA7A9		; branch if GOTO
	LDA #TK_THEN		; set THEN token
	JSR SYNCHR		; scan for CHR$(.A), else do syntax error
LA7A9	LDA FAC1+FAC_EXPT	; get FAC1 exponent
	BEQ LA7B5		; branch if FAC1 = $00
	JSR CHRGOT		; scan memory
	BCS LA76C		; branch if not numeric character, is variable or keyword
	JMP GOTO		; perform GOTO
LA7B5	JMP REM			; REM the rest of the line
#endif ; EXTRA_CMDS


;***********************************************************************************;
;
; perform IEEE

IEEE
	BNE LA80F		; exit if following byte to allow syntax error
	TAY
	DEY
	STY DSFA		; set disk status device number, no device
	LDA #$40		; set IEEE bus
	JMP SETBUS		; set bus mapping


;***********************************************************************************;
;
; perform SERIAL

SERIAL
	BNE LA80F		; exit if following byte to allow syntax error
	TAY
	DEY
	STY DSFA		; set disk status device number, no device
	JMP SETBUS		; set bus mapping


;***********************************************************************************;
;
; parse file name and, optionally, device and drive number

PARSEFNAME
	JSR PARSEPARAM		; parse disk command parameters
	AND #$E6		; mask illegal parameters
	BNE LA80F		; if any set do syntax error
LA7D9	LDA PARSEST		; get parse state
	LSR			; shift primary file name flag into Cb
	BCC LA80F		; if file name not present do syntax error
	ROL			; restore parse state
	RTS

LA7E0	AND #$C4		; mask illegal parameters
	BNE LA80F		; if any set do syntax error
	LDA PARSEST		; get parse state
LA7E6	AND #$03		; mask primary and secondary file name
	CMP #$03		; test both are present
LA7EA	BNE LA80F		; if either missing do syntax error
	LDA PARSEST		; get parse state
	RTS


;***********************************************************************************;
;
; parse file name and drive number

PARSENAMD
	JSR PARSEPARAM		; parse disk command parameters
	AND #$05		; mask primary drive number and file name
	CMP #$05		; test both are present
	JMP LA7EA		; do syntax error if either are missing


;***********************************************************************************;
;
; set secondary address to free value

SETFREESA
	LDY #$61		; set secondary address to $01
LA7FB	INY			; increment secondary address
	TYA			; copy secondary address
	LDX LDTND		; get open file count
LA7FF	DEX			; decrement count to give index
	BMI LA809		; exit if no files
	CMP SAT,X		; compare with secondary address table
	BEQ LA7FB		; loop if secondary address found
	BNE LA7FF		; loop through whole table
LA809	STY SA			; set secondary address
	RTS

LA80C	JMP LABE5		; do illegal quantity error
LA80F	JMP $CF08		; do syntax error then warm start


;***********************************************************************************;
;
; perform RECORD

RECORD
	LDA #$01		; set byte position to first byte
	STA CMDBUF+4		; set record byte position
	JSR CHRGOT		; scan memory
	LDA #'#'		; get # token
	JSR SYNCHR		; scan for CHR$(.A), else do syntax error
	JSR LAC36		; evaluate byte expression, result in .X
	TXA			; copy logical file
	BEQ LA80C		; if = 0 do illegal quantity error
	STX LA			; set logical file
	JSR COMCHK		; scan for ",", else do syntax error
	BEQ LA80F		; if no following value do syntax error
	BCC LA83D		; branch if numeric
	JSR LPACHK		; scan for "(", else do syntax error
	JSR FRMEVL		; evaluate expression
	JSR MAKADR		; convert FAC1 to integer in temporary integer
	JSR RPACHK		; scan for ")", else do syntax error
	JMP LA843		; branch always

LA83D	JSR FRMEVL		; evaluate expression
	JSR MAKADR		; convert FAC1 to integer in temporary integer
LA843	JSR CHRGOT		; scan memory
	BEQ LA85E		; branch if no more chrs
	JSR COMCHK		; scan for ",", else do syntax error
	BEQ LA80F		; if no following value do syntax error
	JSR LAC36		; evaluate byte expression, result in .X
	TXA			; copy record byte position
	BEQ LA80C		; if = 0 do illegal quantity error
	INX			; increment to test ..
	BEQ LA80C		; if = $FF do illegal quantity error
	STA CMDBUF+4		; set record byte position
	JSR CHRGOT		; scan memory
	BNE LA80F		; branch if not null
LA85E	LDA LA			; get logical file
	JSR $F3D4		; find file .A
	BEQ LA86A		; branch if the file is found
	LDX #ER_FNOTOPEN	; file not found
	JMP ERROR		; do error #.X then warm start

LA86A	JSR SETFLCH		; set file details from table,.X
	LDA #'P'		; "P" (record position)
	STA CMDBUF		; set record position command
	LDA SA			; get secondary address
	STA CMDBUF+1		; set channel number
	LDA LINNUM		; get temporary integer low byte
	STA CMDBUF+2		; set record number low byte
	LDA LINNUM+1		; get temporary integer high byte
	STA CMDBUF+3		; set record number high byte
	LDX #$05		; set command length
	JSR LAA69		; setup buffer address and length
	JMP LA954		; send on command channel


;***********************************************************************************;
;
; Prompt the user to proceed with a destructive action. If negative then the calling
; subroutine is terminated.

RUSURE
	LDA CHRGOT+2		; save BASIC execute pointer high byte
	CMP #$02		; compare with $02xx
	BNE LA8AC		; if not immediate mode return
	LDY #$0D		; set index to length of prompt
LA891	LDA LA8AD,Y		; get character from prompt
	JSR SCRNOUT		; output character
	DEY			; decrement index
	BPL LA891		; loop until all done
	JSR XCHRIN		; get input from keyboard
	PHA			; save key
LA89E	JSR XCHRIN		; get input from keyboard
	CMP #$0D		; compare with [CR]
	BNE LA89E		; loop until [CR]
	PLA			; restore first key
	CMP #'Y'		; compare key with "Y"
	BEQ LA8AC		; branch if accepted
	PLA			; dump return address low byte
	PLA			; dump return address high byte
LA8AC	RTS

LA8AD	.BYTE "?ERUS UOY ERA",$0D


;***********************************************************************************;
;
; close all logical files on the current device

CLOSEALL
	LDA FA			; get device number
	LDX LDTND		; get open file count
LA8BF	DEX			; decrement count to give index
	BMI LA8AC		; exit if no files
	CMP FAT,X		; compare device number with table device number
	BNE LA8BF		; loop if no match
	LDA LAT,X		; get logical file number from table logical file number
	JSR XCLOSE		; close a specified logical file
	JMP CLOSEALL		; loop for all logical files


;***********************************************************************************;
;
; perform DCLOSE

DCLOSE
	JSR PARSEPARAM		; parse disk command parameters
	AND #$F3		; mask illegal parameters
	BNE LA941		; if any set do syntax error
	LDA LA			; get logical file
	BEQ CLOSEALL		; if no logical file close all logical files
	JMP XCLOSE		; close specific logical file


;***********************************************************************************;
;
; perform DOPEN

DOPEN
	JSR PARSENAMD		; parse file name and drive number
	AND #$22		; mask illegal parameters
	BNE LA906		; if any set do syntax error
	JSR SETFREESA		; set secondary address to free value
	LDX #$00
	BIT PARSEST		; test parse state
	BPL LA8F4		; branch if replace flag not set
	LDA #'@'		; set replace file
	STA CMDBUF		; store in file name
	INX			; increment file name length
LA8F4	LDY #$03		; set template index
	TYA			; set template length
	BVC LA8FB		; branch if write flag not set
	LDA #$07		; set template length
LA8FB	JSR LAA06		; build disk command
	JMP $E1BE		; open a logical file, handle I/O error


;***********************************************************************************;
;
; perform APPEND

APPEND
	JSR PARSENAMD		; parse file name and drive number
	AND #$E2		; mask illegal parameters
LA906	BNE LA923		; if any set do syntax error
	JSR SETFREESA		; set secondary address to free value
	LDY #$0A		; set template index
	LDA #$05		; set template length
	INX
	BEQ LA8FB		; build disk command & open logical file, branch always


;***********************************************************************************;
;
; perform SCRATCH

SCRATCH
	JSR PARSEFNAME		; parse file name etc
	JSR RUSURE		; prompt for confirmation
	LDY #$37		; set template index
	BNE LA94F		; perform disk command, branch always


;***********************************************************************************;
;
; perform HEADER

HEADER
	JSR PARSEFNAME		; parse disk name etc
	AND #$11		; mask mandatory parameters, primary drive number & file name
	CMP #$11		; test all are present
LA923	BNE LA941		; if any missing do syntax error

	JSR CLOSEALL		; close all logical files
	JSR RUSURE		; prompt for confirmation
	LDY #$0F		; set template index
	LDA #$04		; set template length
	LDX DISKID		; get first disk identifier character
	BEQ LA951		; if not set perform disk command
	LDA #$06		; set template length
	BNE LA951		; perform disk command, branch always


;***********************************************************************************;
;
; perform BACKUP

BACKUP
	JSR PARSEPARAM		; parse disk command parameters
	AND #$30		; mask mandatory parameters, primary & secondary drive number
	CMP #$30		; test all are present
	BEQ LA944		; if all present continue
LA941	JMP $CF08		; do syntax error then warm start

LA944	LDA PARSEST		; get parse state
	AND #$C7		; mask illegal parameters
	BNE LA941		; if any set do syntax error
	JSR CLOSEALL		; close all logical files
	LDY #$17		; set template index
LA94F	LDA #$04		; set template length
LA951	JSR LAA04		; build disk command
LA954	LDA #$6F		; set secondary address to $0F
	STA SA			; set secondary address
	JSR XOPNCHN		; open logical channel
	RTS


;***********************************************************************************;
;
; perform COPY

COPY
	JSR PARSEPARAM		; parse disk command parameters
	AND #$30		; mask primary & secondary drive number
	CMP #$30		; test both are present
	BNE LA96B		; branch if both not present
	LDA PARSEST		; get parse state
	AND #$C7		; mask illegal parameters
	BEQ LA970

LA96B	LDA PARSEST		; get parse state
	JSR LA7E0		; test for illegal parameters
LA970	LDY #$1B		; set template index
LA972	LDA #$08		; set template length
	BNE LA951		; branch always


;***********************************************************************************;
;
; perform CONCAT

CONCAT
	JSR PARSEPARAM		; parse disk command parameters
	JSR LA7E0		; test for illegal parameters
	LDY #$23		; set template index
	LDA #$0C		; set template length
	BNE LA951		; branch always


;***********************************************************************************;
;
; perform CD

CD
	JSR PARSEPARAM		; parse disk command parameters
	AND #$F6		; mask illegal parameters
	BNE LA941		; if any set do syntax error
	LDA PARSEST		; get parse state
	AND #$01		; mask primary file name
	BEQ LA941		; if primary file name missing do syntax error
	LDY #$3B		; set template index
	LDA #$04		; set template length
	BNE LA951		; branch always


;***********************************************************************************;
;
; perform COLLECT

COLLECT
	JSR PARSEPARAM		; parse disk command parameters
	AND #$E7		; mask illegal parameters
	BNE LA941		; if any set do syntax error
	JSR CLOSEALL		; close all logical files
	LDY #$15		; set template index
	LDX #$01		; set template length
	LDA PARSEST		; get parse state
	AND #$10		; mask primary drive number
	BEQ LA997		; branch if not set
	INX			; increment template length
LA997	TXA			; set template length
	BNE LA951		; branch always


;***********************************************************************************;
;
; perform RENAME

RENAME
	JSR PARSEPARAM		; parse disk command parameters
	JSR LA7E6		; test primary and secondary file name are present
	AND #$E4		; mask illegal parameters
LA9A2	BNE LA941		; if any set do syntax error
	LDY #$2F		; set template index
	BNE LA972		; branch always


;***********************************************************************************;
;
; perform DLOAD

DLOAD
	JSR PARSEPARAM		; parse disk command parameters
	JSR LA7D9		; test primary file name is present
	AND #$E6		; mask illegal parameters
	BNE LA9A2		; if any set do syntax error
	LDY #$03		; set template index
	TYA			; set template length
	JSR LAA04		; build disk command
	LDX #$00		; flag load
	STX VERCHK		; set load/verify flag
	STX SA			; set secondary address
	JMP $E16C		; jump to BASIC load


;***********************************************************************************;
;
; perform DSAVE

DSAVE
	JSR PARSEPARAM		; parse disk command parameters
	JSR LA7D9		; test primary file name is present
	AND #$66		; mask illegal parameters
	BNE LA9A2		; if any set do syntax error
	TAX			; clear .X
	LDY #$03		; set template index
	BIT PARSEST		; test parse state
	BPL LA9D8		; branch if replace flag not set
	LDA #'@'		; set replace file
	STA CMDBUF		; store in file name
	INX			; increment file name length
LA9D8	TYA			; set template length
	JSR LAA06		; build disk command
	JMP $E156		; jump to BASIC save


;***********************************************************************************;
;
; copy primary file name to command buffer

COPYFNAME1
	LDA FNLEN		; get file name length
	STA CMP0		; set length
	LDA FNADR		; get file name address low byte
	STA EAL			; set address low byte
	LDA FNADR+1		; get file name address high byte
	STA EAL+1		; set address high byte


;***********************************************************************************;
;
; copy secondary file name to command buffer

COPYFNAME2
	TYA			; copy .Y
	PHA			; save .Y
	LDY CMP0		; get file name length
	BEQ LA9FF		; exit if zero length
	LDY #$00		; clear index
LA9F3	LDA (EAL),Y		; get character
	STA CMDBUF,X		; set character
	INX			; increment destination index
	INY			; increment source index
	CPY CMP0		; compare source index with length
	BNE LA9F3		; loop while not equal
	.BYTE $24		; makes next line BIT $CA
LA9FF	DEX			; decrement destination index
	PLA			; pull .Y
	TAY			; restore .Y
	SEC
	RTS


;***********************************************************************************;
;
; build disk command

LAA04	LDX #$00		; set command length
LAA06	STA TEMPLEN		; set template length
LAA09	DEC TEMPLEN		; decrement template length
	BMI LAA69		; if non-zero setup buffer address and length
	INY			; increment index
	LDA CMDTEMPL,Y		; get template character
	BPL LAA63		; if normal character store in buffer

	CMP #$F1		; compare with primary file name token
	BNE LAA1B		; branch if no match
	JSR COPYFNAME1		; copy primary file name to buffer

LAA1B	CMP #$F2		; compare with secondary file name token
	BNE LAA22		; branch if no match
	JSR COPYFNAME2		; copy primary file name to buffer

LAA22	CMP #$E0		; compare with record length token
	BNE LAA2B		; branch if no match
	LDA RECLEN		; get record length
	BNE LAA63		; store in command, branch always

LAA2B	CMP #$D0		; compare with disk name token
	BNE LAA3E		; branch if no match
	LDA DISKID		; get first disk identifier character
	STA CMDBUF,X		; store in command
	INX			; increment index
	LDA DISKID+1		; get second disk identifier character
	STA CMDBUF,X		; store in command
	INX			; increment index
	TXA			; copy index to .A

LAA3E	CMP #$E1		; compare with record length/write
	BNE LAA53		; branch if no match
	LDA RECLEN		; get record length
	BNE LAA4F		; if set store length indicator
	LDA #'S'		; SEQ file type
	STA RECLEN
	LDA #'W'		; write file mode
	.BYTE $2C		; makes next line BIT $4CA9
LAA4F	LDA #'L'		; "L", record length
	BCS LAA63		; store in command, branch always

LAA53	CMP #$D1		; compare with primary drive number
	BNE LAA5B		; branch if no match
	LDA DRIVENO1		; get primary drive number
	BPL LAA61		; branch always

LAA5B	CMP #$D2		; compare with secondary drive number
	BNE LAA09		; if no match process next template token
	LDA DRIVENO2		; get secondary drive number
LAA61	ORA #'0'		; convert to numeric character
LAA63	STA CMDBUF,X		; store in command
	INX			; increment index
	BNE LAA09		; branch always

LAA69	STX FNLEN		; set buffer length
	LDA #<CMDBUF		; get command buffer low byte
	STA FNADR		; set buffer pointer low byte
	LDA #>CMDBUF		; get command buffer high byte
	STA FNADR+1		; set buffer pointer high byte
	RTS


;***********************************************************************************;
;
; parse disk command parameters

PARSEPARAM
	LDX #$00		; clear .X
	STX PARSEST		; clear parse state
	STX LA			; clear logical file
	STX RECLEN		; clear record length/disk identifier flag
	STX DRIVENO2		; clear secondary drive number
	STX DRIVENO1		; clear primary drive number
	STX FNLEN		; clear primary file name length
	STX CMP0		; clear secondary file name length
	STX DISKID		; clear first disk identifier character
	STX STATUS		; clear I/O status byte
	LDX #$08		; set device to 8, first disk
	STX FA			; set device number
	JSR CHRGOT		; scan memory
	BEQ LAAFA		; if no more chrs get parse state and exit

; parse primary parameters

LAA93	CMP #'#'		; compare with "#" character
	BEQ EVALLFN		; evaluate logical file parameter
	CMP #'W'
	BEQ EVALRECW		; evaluate record length/write parameter
	CMP #'L'		; compare with "L", record length
	BEQ EVALRECW		; evaluate record length/write parameter
	CMP #'R'
	BNE LAAA9
LAAA3	JSR CHRGET		; increment and scan memory
LAAA6	JMP LAB6B

LAAA9	CMP #'D'		; compare with "D", drive number
	BEQ LAB08		; evaluate primary drive number parameter
	CMP #TK_ON		; compare with ON token
	BEQ LAAFE		; evaluate device number parameter
	CMP #'U'		; compare with "U", device number
	BEQ LAB03		; evaluate device number parameter
	CMP #'I'		; compare with "I", disk identifier
	BEQ LAB1D		; evaluate disk identifier
	CMP #$22		; compare with quote character
	BEQ LAAFC		; evaluate primary file name parameter
	CMP #'('		; compare with "("
	BEQ LAAFC		; evaluate primary file name parameter
LAAC1	JMP $CF08		; do syntax error then warm start


;***********************************************************************************;
;
; evaluate logical file parameter

EVALLFN
	LDA PARSEST		; get parse state
	AND #$04		; mask logical file bit
	BNE LAAC1		; if already set do syntax error
	JSR GETNUMERIC		; get byte expression in .X
	TXA			; copy value to .A
	BEQ LAAD6		; if zero do illegal quantity error
	STX LA			; set logical file
	LDA #$04		; set logical file bit
	BNE LAAF5		; update parse state, branch always

LAAD6	JMP LABE5		; do illegal quantity error


;***********************************************************************************;
;
; evaluate record length/write parameter

EVALRECW
	BIT PARSEST		; test parse state
	BVS LAAC1		; if record length/write parsed then do syntax error
	CMP #'W'		; compare with "W"
	BNE LAAE7		; if not "W" then evaluate record length parameter
	JSR CHRGET		; increment and scan memory
	JMP LAAF3		; set record length/write bit in parse state and continue


;***********************************************************************************;
;
; evaluate record length parameter

LAAE7	JSR GETNUMERIC		; get byte expression in .X
	TXA			; copy value to .A
	BEQ LAAD6		; if zero do illegal quantity error
	INX			; increment to roll over to zero
	BEQ LAAD6		; if was 255 do illegal quantity error
	STA RECLEN		; save record length
LAAF3	LDA #$40		; set record length/write bit
LAAF5	JSR LAC00		; update parse state
	BNE LAAA6		; branch always

LAAFA	BEQ LAB70		; get parse state and exit
LAAFC	BEQ LAB3C		; evaluate primary file name parameter

LAAFE	JSR GETDEVNUM		; evaluate device number parameter
	BNE LAB6B		; branch always


;***********************************************************************************;
;
; evaluate device number parameter

LAB03	JSR LABF1
	BNE LAB6B		; branch always


;***********************************************************************************;
;
; evaluate primary drive number parameter

LAB08	LDA PARSEST		; get parse state
	AND #$10		; mask primary drive number bit
	BNE LAAC1		; if set do syntax error
	JSR GETNUMERIC		; get byte expression in .X
	CPX #$02		; compare with max drive + 1
	BCS LAAD6		; if >= do illegal quantity error
	STX DRIVENO1		; set primary drive number
	STX DRIVENO2		; set secondary drive number
	LDA #$10		; set primary drive number parameter bit
	BNE LAAF5		; update parse state, branch always


;***********************************************************************************;
;
; evaluate disk identifier parameter

LAB1D	LDA RECLEN		; get record length/disk identifier flag
	BNE LAAC1		; if either already parsed do syntax error
	JSR CHRGET		; increment and scan memory
	BEQ LAB9C		; branch if no more chrs
	STA DISKID		; set first disk identifier character
	JSR CHRGET		; increment and scan memory
	BEQ LAB9C		; branch if no more chrs
	STA DISKID+1		; set second disk identifier character
	LDA #$FF		; set invalid record length
	STA RECLEN		; set record length/disk identifier flag
	JMP LAAA3		; continue parsing

LAB3A	BEQ LAAFE		; evaluate device number parameter, branch always


;***********************************************************************************;
;
; evaluate primary file name parameter

LAB3C	LDA PARSEST		; get parse state
	LSR			; shift primary file name bit into Cb
	JSR GETSTRING		; evaluate string
	STA FNLEN		; set file name length
	STA TEMPLEN		; set string length
	LDA #<TEMPSTR
	STA FNADR		; set file name pointer low byte
	LDA #>TEMPSTR
	STA FNADR+1		; set file name pointer high byte
	LDY #$00		; clear index
	LDA (INDEX),Y		; get string character
	CMP #'@'		; compare with replace command
	BNE LAB5B		; branch if not replace
	INC FNADR		; increment file name pointer low byte
	DEC FNLEN		; decrement file name length
LAB5B	STA TEMPSTR,Y		; store string character
	INY			; increment index
	LDA (INDEX),Y		; get string character
	CPY TEMPLEN		; compare with string length
	BCC LAB5B		; loop until equal
	LDA #$01		; set primary file name bit
	JSR LAC00		; update parse state
LAB6B	JSR CHRGOT		; scan memory
	BNE LAB73		; branch if not null
LAB70	LDA PARSEST		; get parse state
	RTS

LAB73	CMP #','
	BNE LAB7D
	JSR CHRGET		; increment and scan memory
	JMP LAA93		; continue parsing primary parameters


;***********************************************************************************;
;
; parse secondary parameters

LAB7D	CMP #TK_ON		; compare with ON token
	BEQ LAB3A		; evaluate device number parameter
	CMP #TK_TO		; compare with TO token
	BNE LAB9C		; if not TO do syntax error
LAB85	JSR CHRGET		; increment and scan memory
	CMP #'D'		; compare with "D", drive number
	BEQ LAB9F		; evaluate secondary drive number parameter
	CMP #TK_ON		; compare with ON token
	BEQ LABB5		; if ON get device number
	CMP #'W'		; compare with "W"
	BEQ LABBA
	CMP #$22		; compare with quote character
	BEQ LABBF		; evaluate secondary file name parameter
	CMP #'('		; compare with "("
	BEQ LABBF		; evaluate secondary file name parameter
LAB9C	JMP $CF08		; do syntax error then warm start


;***********************************************************************************;
;
; evaluate secondary drive number parameter

LAB9F	LDA PARSEST		; get parse state
	AND #$20		; mask secondary drive number bit
	BNE LAB9C		; if set do syntax error
	JSR GETNUMERIC		; get byte expression in .X
	CPX #$02		; compare with max drive + 1
	BCS LABE5		; if >= do illegal quantity error
	STX DRIVENO2		; set secondary drive number
	LDA #$20		; set secondary drive number bit
	JSR LAC00		; update parse state
	BNE LABD1		; continue parsing, branch always

LABB5	JSR GETDEVNUM		; evaluate device number parameter
	BNE LABD1		; continue parsing, branch always

LABBA	JSR LABF1
	BNE LABD1		; continue parsing, branch always


;***********************************************************************************;
;
; evaluate secondary file name parameter

LABBF	LDA PARSEST		; get parse state
	LSR			; shift secondary file name bit ..
	LSR			; .. into Cb
	JSR GETSTRING		; evaluate string
	STA CMP0		; set string length
	STX EAL			; set string pointer low byte
	STY EAL+1		; set string pointer high byte
	LDA #$02		; set secondary file name bit
	JSR LAC00		; update parse state
LABD1	JSR CHRGOT		; scan memory
	BEQ LAB70		; if no more chrs get parse state and exit
	CMP #','		; compare with ","
	BEQ LAB85
	CMP #TK_ON		; compare with ON token
	BEQ LABB5		; if ON get device number
	CMP #'W'		; compare with "W"
	BEQ LABBA
LABE2	JMP $CF08		; do syntax error then warm start

LABE5	LDX #ER_ILLQUAN		; illegal quantity error
LABE7	JMP ERROR		; do error #.X then warm start


;***********************************************************************************;
;
; evaluate device number parameter

GETDEVNUM
	JSR CHRGET		; increment and scan memory
	CMP #'U'		; compare with "U", device number parameter
LABEF	BNE LABE2		; if not "U" do syntax error
LABF1	JSR GETNUMERIC		; get byte expression in .X
	CPX #$60		; compare with max device + 1
	BCS LABE5		; if device number too large do illegal quantity error
	CPX #$04		; compare with first device number
	BCC LABE5		; if too small do illegal quantity error
	STX FA			; set device number
	LDA #$08		; set device number bit
LAC00	ORA PARSEST		; OR in new bit
	STA PARSEST		; set parse state
	RTS


;***********************************************************************************;
;
; evaluate string, returns with .A = length, .X = pointer low byte, .Y = pointer high byte

GETSTRING
	BCS LABE2		; if already parsed do syntax error
	JSR FRMEVL		; evaluate expression
	JSR DELST		; evaluate string
	TAX			; copy length to .X
	BEQ LABE5		; if empty do illegal quantity error
	CMP #$12
	BCS LAC20		; if >= do string too long
	LDY #$00		; clear index
	LDA (INDEX),Y		; get first character of string
	CMP #'@'		; compare with replace command
	BEQ LAC24		; branch if replace
	CPX #$11
	BCC LAC2D		; branch if length ok
LAC20	LDX #ER_STR2LONG	; string too long
	BNE LABE7		; do error #.X then warm start
LAC24	LDA PARSEST		; get parse state
	BMI LABE2		; do error if replace already processed
	LDA #$80		; set replace bit
	JSR LAC00		; update parse state
LAC2D	TXA			; copy length to .A
	LDX INDEX		; get string pointer low byte
	LDY INDEX+1		; get string pointer high byte
	RTS


;***********************************************************************************;
;
; evaluate byte expression, result in .X

GETNUMERIC
	JSR CHRGET		; increment and scan memory
LAC36	BEQ LABE2		; if no more chrs do syntax error
	BCC LAC43		; branch if numeric
	JSR LPACHK		; scan for "(", else do syntax error
	JSR $D79E		; get byte parameter
	JMP RPACHK		; scan for ")", else do syntax error

LAC43	JMP $D79E		; get byte parameter
LAC46	JMP LAD1F		; close logical file and restore previous output device

	
;***********************************************************************************;
;
; perform CATALOG/DIRECTORY

CATALOG
	LDA LA			; get logical file
	STA EAL+1		; save current logical file
	JSR PARSEPARAM		; parse disk command parameters
	AND #$E6		; mask illegal parameters
	BNE LABEF		; if any set do syntax error
	STA CMP0		; clear header/directory entry flag
	LDY #$FF		; set template index
	LDX #$01		; set template length
	LDA PARSEST		; get parse state
	LSR			; shift primary file name bit into Cb
	BCS LAC65		; branch if file name given
	AND #$08		; test primary drive number bit ($10 shifted right)
	BEQ LAC67		; branch if no drive number given
	INX			; increment template length
	.BYTE $2C		; makes next line BIT $04A2
LAC65	LDX #$04
LAC67	TXA			; set template lenth
	JSR LAA04		; build disk command
	LDA DFLTO		; get output device number
	STA EAL			; save current output device
	LDA #$60		; set secondary address to $00
	STA SA			; set secondary address
	STA LA			; set logical file
	JSR XOPEN		; open a logical file
	LDA STATUS		; get I/O status byte
	BEQ LAC81		; branch if no error
	LDX #ER_DEVNOTP		; device not present
	JMP ERROR		; do error #.X then warm start

LAC81	LDY #$03		; consume 3 words, load address, line link, line number

LAC86	LDX #$60		; set logical file to $60
	JSR XCHKIN		; open a channel for input
LAC8B	JSR XACPTR		; get number of blocks low byte
	STA FAC1+FAC_MANT+1	; save byte
	LDX STATUS		; get I/O status byte
	BNE LAC46		; if any error close logical file and restore previous output device
	JSR XACPTR		; get number of blocks high byte
	STA FAC1+FAC_MANT	; save byte
	DEY			; decrement word count
	BNE LAC8B		; loop until done
	JSR XCLRCHN		; close input and output channels
	JSR RESTOUT		; restore previous output device
	LDX FAC1+FAC_MANT+1
	JSR $DDD1		; print number of blocks as unsigned integer
	JSR PRINTSP		; print space
	JSR XCLRCHN		; close input and output channels
	LDA #$12		; file name + two quotes
	STA FNLEN		; set file name length
	LDX EAL			; get saved output device
	CPX #$03		; compare with screen
	BNE LACC0		; if not screen skip column check
	LDA PNTR		; get cursor column
	CMP #$04		; compare with column 4
	BCS LACC0		; if beyond column 3 already skip column reset
	DEX			; decrement column to 2
	STX PNTR		; set cursor column

LACC0	LDX #$60		; set logical file to $60
	JSR XCHKIN		; open a channel for input
	JSR XACPTR		; input a byte from either the serial or IEEE bus
	PHA			; save byte
	JSR XCLRCHN		; close input and output channels
	LDA STATUS		; get I/O status byte
	BNE LAD1E		; if any error close and exit
	PLA			; restore byte
	BEQ LAD14		; if EOL print [CR] and continue
	LDX EAL			; get saved output device
	CPX #$03		; compare with screen
	BNE LAD00		; if not screen skip entry transformation
	CMP #' '		; compare with space
	BNE LACE3		; if not space it is the file name
	LDX PNTR		; get cursor column
	CPX #$02
	BEQ LACC0		; if padding at start of line then skip
LACE3	LDX FNLEN		; get file name length
	BMI LACC0		; beyond file name, skip rest of entry
	BNE LACFE		; part of file name, print it
	LDX CMP0		; get header/directory entry flag
	BEQ LAD00		; if in header just print it
	CMP #' '		; compare with space
	BEQ LACC0		; gap following file name, skip
	AND #$3F		; mask top two bits
	LDY #$15		; set column count to 21
	STA (PNT),Y		; save file type character to current screen line
	LDA COLOR		; get current colour code
	STA (USER),Y		; save to colour RAM
	LDA #':'
LACFE	DEC FNLEN		; decrement file name length
LAD00	JSR LA1D2		; send character to saved output device
	JSR STOP		; scan stop key
	BEQ LAD1F		; if stop close logical file and restore previous output device
	JSR XGETIN		; get character from keyboard queue
	BEQ LACC0		; if no key pressed continue
LAD0D	JSR XGETIN		; get character from keyboard queue
	BEQ LAD0D		; loop until key pressed
	BNE LACC0		; branch always

; end of line reached

LAD14	JSR LA1D0		; output CR to saved output device
	LDY #$02		; consume 2 words, line link, line number
	STY CMP0		; set header/directory entry flag
	JMP LAC86		; continue with next entry

LAD1E	PLA			; pop saved byte
LAD1F	JSR XCLRCHN		; close input and output channels
	LDA #$60		; set logical file to $60
	JSR XCLOSE		; close a specified logical file
	JMP RESTOUT		; restore previous output device


;***********************************************************************************;
;
; output space

PRINTSP
	LDA #' '
	JMP XCHROUT		; output character to channel


#ifdef EXTRA_CMDS
;***********************************************************************************;
;
; get non-space character

GETNOTSP
	JSR LAD38		; get character
	CMP #' '		; compare with space
	BEQ GETNOTSP		; loop if space
	RTS


;***********************************************************************************;
;
; get character and compare with [CR], return Zb = 1 if so

GETCR	JSR XCHRIN		; input character from channel
	CMP #$0D		; compare with [CR]
	RTS

LAD38	JSR GETCR		; get character and compare with [CR]
	BNE LAD6F		; return if not [CR]
	PLA			; else dump calling address low byte
	PLA			; dump calling address high byte
	BCS MONPROMPT		; if >= [CR] display prompt
LAD41	DEX			; decrement table index
	BPL LAD5F		; if >= 0 continue else ..


;***********************************************************************************;
;
; handle syntax error

MONERR
	LDA #'?'
	JSR XCHROUT		; output character to channel
	LDX #$F8		; set value for stack pointer
	TXS			; set stack then ..


;***********************************************************************************;
;
; display prompt

MONPROMPT
	LDA #$0D		; [CR]
	LDX #'.'
	JSR PRINTAX		; output prompt on new line
LAD56	JSR GETNOTSP		; get non-space character
	CMP #'.'
	BEQ LAD56		; loop while dot
	LDX #$07		; set number of table entries
LAD5F	CMP MONKEYS,X		; compare with table entry
	BNE LAD41
	EOR #'L'
	STA TEMPST+4		; save load/save flag
	LDA MONCMDSH,X		; set command execute pointer high byte
	PHA			; push it
	LDA MONCMDSL,X		; get command execute pointer low byte
	PHA			; push it
LAD6F	RTS			; execute command


;***********************************************************************************;
;
; print 16 bit value as hex

PRINTHEX16
	LDA STAL+1		; get high byte
	JSR PRINTHEX8		; output as hex digits
	LDA STAL		; get low byte
PRINTHEX8
	PHA			; save value
	LSR			; shift top nibble ..
	LSR
	LSR
	LSR			; .. into bottom nibble
	JSR LAD82		; output most significant nibble as hex digit
	PLA			; restore value
	AND #$0F		; mask off top nibble
LAD82	ORA #'0'
	CMP #'9'+1
	BCC LAD8A		; numeric, output it
	ADC #$06		; make hex digit
LAD8A	JMP XCHROUT		; output character to channel


;***********************************************************************************;
;
; display .A, .X

PRINTAX
	JSR XCHROUT		; output character to channel
	TXA			; copy second character to .A
	.BYTE $2C		; makes next line BIT $0DA9


;***********************************************************************************;
;
; output [CR]

PRINTCR
	LDA #$0D
	JMP XCHROUT		; output character to channel


;***********************************************************************************;
;
; get 8 bit hex value

GETHEX8
	LDA #$00		; clear .A
	STA TEMPST+3		; clear value
	JSR GETNOTSP		; get non-space character
	JSR LADA7		; convert first digit
	JSR LAD38		; get character
LADA7	CMP #'0'		; compare with "0"
	BCC MONERR		; if < do syntax error
	CMP #'9'+1
	AND #$0F		; convert to hex nibble
	BCC LADB3		; if a decimal digit skip
	ADC #'A'-'9'		; else adjust alphabetic
LADB3	ASL TEMPST+3		; shift value ..
	ASL TEMPST+3
	ASL TEMPST+3
	ASL TEMPST+3		; .. into high nibble
	ORA TEMPST+3		; OR in low nibble
	STA TEMPST+3		; save value
	SEC
LADC0	RTS


;***********************************************************************************;
;
; scan stop key, close input and output channels if pressed

CHKSTOP
	JSR STOP		; scan stop key
	BNE LADC0		; exit if no stop
	PHP			; save status
	JSR XCLRCHN		; close input and output channels
	PLP			; restore status
	RTS

LADCC	LDX #TEMPST
	.BYTE $2C		; makes next line BIT $C1A2
LADCF	LDX #STAL
LADD1	JSR GETHEX8		; get 8 bit hex value
	STA $01,X		; set high byte
	JSR GETHEX8		; get 8 bit hex value
	STA $00,X		; set low byte
	RTS

LADDC	INC STAL		; increment low byte
	BNE LADE2		; if no rollover skip the high byte increment
	INC STAL+1		; increment high byte
LADE2	RTS

LADE3	LDA #<SAVEREG		; get saved registers low byte
	STA STAL		; set start address low byte
	LDA #>SAVEREG		; get saved registers high byte
	STA STAL+1		; set start address high byte
	LDA #$05		; number of registers
	RTS


;***********************************************************************************;
;
; print memory region as hex, STAL = start address, .A = length

PRINTMEM
	STA TEMPST+2		; set byte count
	LDY #$00		; clear index
LADF2	JSR PRINTSP		; print space
	LDA (STAL),Y		; get byte
	JSR PRINTHEX8		; print it as hex
	JSR LADDC		; increment start address pointer
	BEQ LAE04		; if high byte rolled over exit
	DEC TEMPST+2		; decrement byte count
	BNE LADF2		; loop until zero
	INY
LAE04	RTS

MONKEYS
	.BYTE "MRXG:;LS"

REGHEAD
	.BYTE ";.",$0D,"PS RY RX CA RS  CP   ",$0D


;***********************************************************************************;
;
; perform MONITOR dump memory

MONDUMPMEM
	JSR LADCF		; get start address
	LDX #MEMUSS
	JSR LADD1		; get end address

LAE2E	JSR CHKSTOP		; check for [STOP] key
	BEQ LAE98		; branch if [STOP]
	JSR PRINTCR		; print [CR]
	LDA #'.'
	LDX #':'
	JSR PRINTAX		; print memory dump prefix
	JSR PRINTHEX16
	LDA #$05		; set region length
	JSR PRINTMEM		; print memory region as hex
	BEQ LAE98		; if start address rolled over then exit
	LDA MEMUSS		; get end address low byte
	CMP STAL		; compare with start address low byte
	LDA MEMUSS+1		; get end address high byte
	SBC STAL+1		; subtract start address high byte
	BCS LAE2E		; loop until end reached
	BCC LAE98		; display prompt, branch always


;***********************************************************************************;
;
; BRK handler, enter into MONITOR

B4BREAK
	JSR XCLRCHN		; close input and output channels
	PLA			; pull .Y
	STA SAVEREG+3		; set saved .Y
	PLA			; pull .X
	STA SAVEREG+2		; set saved .X
	PLA			; pull .A
	STA SAVEREG+1		; set saved .A
	PLA			; pull status
	STA SAVEREG		; set saved status
	PLA			; pull program counter low byte
	STA TEMPST		; set saved program counter low byte
	PLA			; pull program counter high byte
	STA TEMPST+1		; set saved program counter high byte
	TSX			; copy stack pointer
	STX SAVEREG+4		; set saved stack pointer
	LDA TEMPST		; get saved program counter low byte
	BNE LAE71		; branch if not zero
	DEC TEMPST+1		; decrement saved program counter high byte
LAE71	DEC TEMPST		; decrement saved program counter low byte
	JSR PRINTCR		; print [CR]
	LDA #'B'
	LDX #'*'
	JSR PRINTAX		; print break indicator


;***********************************************************************************;
;
; perform MONITOR dump registers

MONDUMPREG
	LDX #$18		; set length of heading
LAE7F	LDA REGHEAD,X		; get heading character
	JSR XCHROUT		; print heading character
	DEX			; decrement index
	BPL LAE7F		; loop while more characters
	LDA TEMPST+1		; get saved program counter high byte
	JSR PRINTHEX8		; print as hex
	LDA TEMPST		; get saved program counter low byte
	JSR PRINTHEX8		; print as hex
	JSR LADE3		; set start address to saved registers
	JSR PRINTMEM		; print saved registers as hex
LAE98	JMP MONPROMPT		; display prompt


;***********************************************************************************;
;
; perform MONITOR set memory

MONSETMEM
	JSR LADCF		; get start address
	LDA #$05		; set byte count
	BNE LAEA8		; branch always


;***********************************************************************************;
;
; perform MONITOR set registers

MONSETREG
	JSR LADCC		; get source address
	JSR LADE3		; set destination address to saved registers
LAEA8	STA TEMPST+2		; set byte count
LAEAA	JSR GETHEX8		; get 8 bit hex value
	BCC LAEB7		; ??? dead code ???
	LDX #$00		; clear index
	STA (STAL,X)		; store value
	CMP (STAL,X)		; verify value
	BNE LAF11		; handle syntax error
LAEB7	JSR LADDC		; increment source address
	DEC TEMPST+2		; decrement byte count
	BNE LAEAA		; loop until done
	BEQ LAE98		; display prompt, branch always


;***********************************************************************************;
;
; perform MONITOR GO

MONGO
	JSR GETCR		; get character and compare with [CR]
	BEQ LAEC8		; branch if [CR]
	JSR LADCC		; read program counter
LAEC8	LDX SAVEREG+4		; get saved stack pointer
	TXS			; restore stack pointer
	LDA TEMPST+1		; get saved program counter high byte
	PHA			; store for return
	LDA TEMPST		; get saved program counter low byte
	PHA			; store for return
	LDA SAVEREG		; get saved status
	PHA			; store for return
	LDA SAVEREG+1		; restore .A
	LDX SAVEREG+2		; restore .X
	LDY SAVEREG+3		; restore .Y
	RTI			; restore status and return


;***********************************************************************************;
;
; perform MONITOR EXIT
	
MONEXIT
	JMP (WARMST)		; do BASIC break entry


;***********************************************************************************;
;
; perform MONITOR LOAD/SAVE

MONLDSV	LDY #$02
	STY FNADR+1		; set file name pointer high byte
	DEY
	STY FA			; set device number to 1 (tape)
	STY SA			; set secondary address to 1
	DEY			; clear .Y
	STY FNLEN		; set file name length to zero
	STY STATUS		; clear I/O status byte
	STY VERCK		; clear the load/verify flag
	LDA #$40
	STA FNADR		; set file name pointer low byte
LAEF2	JSR GETCR		; get character and compare with [CR]
	BEQ LAF14		; if [CR] do load
	CMP #' '		; compare with space
	BEQ LAEF2		; loop while space
	CMP #$22		; compare with quote character
	BNE LAF11		; if no file name do syntax error
LAEFF	JSR GETCR		; get character and compare with [CR]
	BEQ LAF14		; if [CR] do load
	CMP #$22		; compare with quote character
	BEQ LAF28		; if end quote check for other parameters
	STA (FNADR),Y		; save file name character
	INC FNLEN		; increment file name length
	INY			; increment index
	CPY #$13
	BNE LAEFF
LAF11	JMP MONERR		; handle syntax error

LAF14	LDA TEMPST+4		; get load/save flag
	BNE LAF11		; if save do syntax error
	LDA #$00		; flag load
	JSR XLOAD		; load RAM from a device
	JSR CHKSTOP		; check for [STOP] key
	LDA STATUS		; read I/O status byte
	AND #$10		; mask 000x 0000, read error
	BNE LAF11		; if read error then exit
	BEQ LAF65		; display prompt, branch always

LAF28	JSR GETCR		; get character and compare with [CR]
	BEQ LAF14		; if [CR] do load
	CMP #','		; compare with ","
	BNE LAF11		; if not "," do syntax error
	JSR GETHEX8		; get 8 bit hex value
	AND #$0F		; mask low nibble
	BEQ LAF11		; if keyboard device do syntax error
	CMP #$03		; compare device number with screen
	BEQ LAF11		; if screen device do syntax error
	STA FA			; set device number
	JSR GETCR		; get character and compare with [CR]
	BEQ LAF14		; if [CR] do load
	CMP #','		; compare with ","
	BNE LAF11		; if not "," do syntax error
	JSR LADCF		; get start address
	JSR XCHRIN		; input character from channel
	CMP #','		; compare with ","
	BNE LAF11		; if not "," do syntax error
	LDX #EAL
	JSR LADD1		; get end address
LAF56	JSR GETCR		; get character and compare with [CR]
	BNE LAF56		; loop if not [CR]
	LDA TEMPST+4		; get load/save flag
	BEQ LAF11		; if load do syntax error
	JSR PRINTCR		; print [CR]
	JSR XSAVE2		; call XKERNAL save routine
LAF65	JMP MONPROMPT		; display prompt

MONCMDSH
	.BYTE >MONDUMPMEM-1	; dump Memory command
	.BYTE >MONDUMPREG-1	; dump Registers command
	.BYTE >MONEXIT-1	; eXit command
	.BYTE >MONGO-1		; Go command
	.BYTE >MONSETMEM-1	; set memory command (:)
	.BYTE >MONSETREG-1	; set registers command (;)
	.BYTE >MONLDSV-1	; Load command
	.BYTE >MONLDSV-1	; Save command

MONCMDSL
	.BYTE <MONDUMPMEM-1	; dump Memory command
	.BYTE <MONDUMPREG-1	; dump Registers command
	.BYTE <MONEXIT-1	; eXit command
	.BYTE <MONGO-1		; Go command
	.BYTE <MONSETMEM-1	; set memory command (:)
	.BYTE <MONSETREG-1	; set registers command (;)
	.BYTE <MONLDSV-1	; Load command
	.BYTE <MONLDSV-1	; Save command


;***********************************************************************************;
;
; perform MODEM

MODEM
	JSR $F118		; input from RS-232 buffer
	JSR RSNXTIN		; get byte from RS-232 buffer
	TAX			; copy byte to .X
	BEQ LAF99		; branch in no data
	AND #$7F		; clear top bit
	CMP #$08		; compare with ASCII BS
	BNE LAF81		; if not BS continue with input translation
	LDA #$14		; replace with PETSCII DEL
LAF81	CMP #"Z"+1		; compare with ASCII "Z" + 1
	BCC LAF89		; if upper case alpha numeric continue with input translation
	CMP #$5E		; compare with ASCII "^"
	BCC LAF96		; if "[", "\" or "]" go display it
LAF89	CMP #"@"		; compare with ASCII "@"
	BCC LAF96		; if not alphabetic go display it
	CMP #$60		; compare with ASCII "`"
	BCC LAF94		; if upper case alphabetic convert and display it
	AND #$5F		; convert to lower case
	.BYTE $2C		; makes next line BIT $8009
LAF94	ORA #$80		; convert to upper case
LAF96	JSR PRT2SCRN		; display received character to the screen
LAF99	LDA SHFLAG		; get keyboard shift/control/C= flag
	AND #$04		; mask [CTRL]
	BEQ LAFA9		; branch if not [CTRL]
LAFA0	JSR $F1F9		; call KERNAL routine to read keyboard
	BEQ LAFA0		; loop until key pressed
	AND #$1F		; mask to translate "A" to ctrl-A
	BPL LAFAE		; branch always
LAFA9	JSR $F1F9		; call KERNAL routine to read keyboard
	BEQ MODEM		; loop until key pressed
LAFAE	PHA			; save key
	LDA M51CDR		; get 6551 pseudo command register
	LSR			; ??? shift unused bit .. ???
	LSR			; .. into Cb
	PLA			; restore key
	BCC LAFBA		; branch if ???
	JSR PRT2SCRN		; display received character to the screen
LAFBA	BMI LAFD0		; if PETSCII graphic symbol go transmit it
	CMP #$14		; compare with PETSCII [DEL]
	BNE LAFC2		; if not DEL continue with output translation
	LDA #$08		; convert to ASCII BS
LAFC2	CMP #'Z'+1		; compare with PETSCII "Z" + 1
	BCC LAFCA		; if upper case alpha numeric continue with output translation
	CMP #$5E		; compare with PETSCII "↑"
	BCC LAFD0		; if "[", "£" or "]" go transmit it
LAFCA	CMP #$40		; compare with PETSCII "@"
	BCC LAFD0		; if not alphabetic go transmit it
	ORA #$60		; convert to lower case
LAFD0	AND #$7F		; mask top bit
	STA PTR1
	JSR RSOUTSAV		; send byte to RS-232 buffer
	JMP MODEM		; loop forever


;***********************************************************************************;
;
; display character to screen at current cursor position

PRT2SCRN
	PHA			; save character
	LDA #' '
	JSR PRINTAT		; print space to screen
	PLA			; restore character
	PHA			; save character
	CMP #$14		; compare with [DEL]
	BNE LAFE8		; branch if not [DEL]
	LDA #$9D		; [CRSR R]
LAFE8	JSR SCRNOUT		; output character and advance cursor
	LDA #$64		; PETSCII graphics char
	JSR PRINTAT		; print cursor to screen
	PLA			; restore saved character
	RTS


;***********************************************************************************;
;
; display character to screen at current cursor position

PRINTAT
	PHA			; save character
	JSR COLORSYN		; calculate pointer to colour RAM
	PLA			; restore character
	LDX COLOR		; get current colour code
	JMP PUTSCRN		; save character and colour to screen @ cursor
#endif ; EXTRA_CMDS


;***********************************************************************************;
;***********************************************************************************;
;
; IEEE bus interface

;***********************************************************************************;
;
; hardware equates

	; cartridge VIA A (U4)
VIAAPB	= $9800			; VIA A DRB
					; bit	function
					; ---	--------
					;  7	ATN in
					;  6	NDAC in
					;  5	NRFD in
					;  4	DAV in
					;  3	EOI in
					;  2	NDAC out
					;  1	NRFD out
					;  0	DAV out

VIAADDRB = $9802		; VIA A DDRB
VIAADDRA = $9803		; VIA A DDRA
VIAAACR	= $980B			; VIA A ACR
					; bit	function
					; ---	--------

VIAAPCR	= $980C			; VIA A PCR
					; bit	function
					; ---	--------

VIAAIFR = $980D			; VIA A IFR
VIAAIER	= $980E			; VIA A IER
VIAAPA2	= $980F			; VIA A DRA, no handshake

	; cartridge VIA B (U5)
VIABPB	= $9810			; VIA B DRB
					; bit	function
					; ---	--------
					;  7	DIO8 in
					;  6	DIO7 in
					;  5	DIO6 in
					;  4	DIO5 in
					;  3	DIO4 in
					;  2	DIO3 in
					;  1	DIO2 in
					;  0	DIO1 in

VIABDDRB = $9812		; VIA B DDRB
VIABDDRA = $9813		; VIA B DDRA
VIABT1CL = $9814		; VIA B T1C_l
VIABT1CH = $9815		; VIA B T1C_h
VIABACR	= $981B			; VIA B ACR
VIABPCR	= $981C			; VIA B PCR
					; bit	function
					; ---	--------
					; 765	CB2 EOI out
					;  4	CB1 SRQ in
					; 321	CA2 ATN out
					;  0

VIABIFR	= $981D			; VIA B IFR
VIABIER	= $981E			; VIA B IER
VIABPA2	= $981F			; VIA B DRA, no handshake
					; bit	function
					; ---	--------
					;  7	DIO8 out
					;  6	DIO7 out
					;  5	DIO6 out
					;  4	DIO5 out
					;  3	DIO4 out
					;  2	DIO3 out
					;  1	DIO2 out
					;  0	DIO1 out

	; KERNAL entrypoints
ORIOST	= $FE6A			; OR into I/O status byte

DEFRFLG	= $9B			; deferred character flag
DEFRCH	= $9C			; deferred character


;***********************************************************************************;
;
; test for IEEE interface

; Attempt to detect presence of VIA B by setting T1 and checking it reports decreasing values.
; Returns Cb set if hardware appears to be present.

IEVIATEST
.(
	LDA VIABACR		; save contents of memory/registers
	PHA
	LDA VIABT1CL
	PHA
	LDA VIABT1CH
	PHA
	LDA #$00		; T1 & T2 1 shot, shift disabled, PA & PB latch disabled
	STA VIABACR		; set VIA B ACR
	LDY #$0C		; set number of iterations
	LDA #$FF
	STA VIABT1CL		; set VIA B T1C_l
	STA VIABT1CH		; set VIA B T1C_h
LOOP
	STA XSAV		; save previous timer value
	LDA VIABT1CL		; get current timer value
	CMP XSAV		; compare with previous value
	BCS OUT			; if >= then abort, no VIA found
	DEY			; decrement index
	BNE LOOP		; loop until done
	PLA			; discard saved contents
	PLA
	PLA
	SEC			; flag found
	RTS
OUT
	PLA			; restore contents of memory
	STA VIABT1CH
	PLA
	STA VIABT1CL
	PLA
	STA VIABACR
	CLC			; flag not found
	RTS
.)


;***********************************************************************************;
;
; initialize cartridge I/O registers

INITIEVIA
	LDA #$00		; T1 & T2 1 shot, shift disabled, PA & PB latch disabled
	STA VIABACR		; set VIA B ACR
	STA VIAAACR		; set VIA A ACR

	LDA #$FF
	STA VIABIFR		; clear pending VIA B interrupts
	STA VIAAIFR		; clear pending VIA A interrupts

	STA VIABPCR		; set VIA B PCR, CB2 high, CB1 +ve edge, CA2 high, CA1 +ve edge
	STA VIAAPCR		; set VIA A PCR (unused)

	LDA #$7F		; disable interrupts, all bits affected
	STA VIABIER		; modify VIA B interrupts
	STA VIAAIER		; modify VIA A interrupts

	LDA #$07		; IIIIIOOO, ATN in, NDAC in, NRFD in, DAV in,
				; EOI in, NDAC out, NRFD out, DAV out
	STA VIAADDRB		; set VIA A DDRB
	STA VIAAPB		; set VIA A DRB outputs high
	LDA #$FF		; all outputs
	STA VIABDDRA		; set VIA B DDRA, DIO1-8
	STA VIABPA2		; set VIA B DRA, all ones
	LDA #$00		; all inputs
	STA VIABDDRB		; set VIA B DDRB, DIO1-8
	STA DEFRFLG		; clear deferred character flag
	RTS


NOTPRES
	PLA
	LDA #$80		; set device not present bit
	JSR ORIOST		; OR into I/O status byte then ..


;***********************************************************************************;
;
; set IEEE ATN high

IECATN
	LDA VIABPCR		; get VIA B PCR
	ORA #$02		; set CA2 high, ATN high
	STA VIABPCR		; save VIA B PCR
	RTS


;***********************************************************************************;
;
; command an IEEE bus device to TALK

; To use this routine the accumulator must first be loaded with a device number
; between 4 and 30. When called this routine converts this device number to a talk
; address. Then this data is transmitted as a command on the IEEE bus.

IETALK
	ORA #$40		; OR with the TALK command
	.BYTE $2C		; makes next line BIT $2009


;***********************************************************************************;
;
; command devices on the IEEE bus to LISTEN

; This routine will command a device on the IEEE bus to receive data. The
; accumulator must be loaded with a device number between 4 and 30 before calling
; this routine. LISTEN convert this to a listen address then transmit this data as
; a command on the IEEE bus. The specified device will then go into listen mode
; and be ready to accept information.

IELISTEN
	ORA #$20		; OR with the LISTEN command
LB33B	PHA			; save byte to send
	LSR DEFRFLG		; clear deferred character flag
	LDA VIABPCR		; get VIA B PCR
	AND #$FD		; set CA2 low, ATN low
	STA VIABPCR		; save VIA B PCR
	LDA VIAAPB		; get VIA A DRB
	AND #$60		; mask NRFD and NDAC in
	CMP #$60		; test both NRFD and NDAC high
	BEQ NOTPRES		; if high do device not present and clear ATN
	PLA			; restore byte to send then ..


;***********************************************************************************;
;
; output byte on IEEE bus

IESEND
.(
	EOR #$FF		; invert it
	STA VIABPA2		; set VIA B DRA
LABEL1	LDA VIAAPB		; get VIA A DRB
	AND #$20		; mask NRFD in
	BEQ LABEL1		; branch if NRFD low
	LDA VIAAPB		; get VIA A DRB
	AND #$FE		; clear DAV out
	STA VIAAPB		; store VIA A DRB
	LDA #$FF		; set for 65,280 cycles
	STA VIABT1CH		; set VIA B T1C_h
LABEL2	LDA VIAAPB		; get VIA A DRB
	AND #$40		; mask NDAC in
	BNE LABEL3		; if NDAC high clear DAV and take data off the bus
	LDA VIABIFR		; get VIA B IFR
	AND #$40		; mask T1 interrupt
	BEQ LABEL2		; loop while timer not expired
	LDA #$01		; set time out write
	STA STATUS		; save I/O status byte
	RTS

LABEL3	LDA VIAAPB		; get VIA B IFR
	ORA #$01		; set DAV out high
	STA VIAAPB		; store VIA A DRB
	LDA #$FF		; set data to $00
	STA VIABPA2		; set VIA B DRA
	RTS
.)


;***********************************************************************************;
;
; command the IEEE bus to UNLISTEN

; This routine commands all devices on the IEEE bus to stop receiving data from
; the computer. Calling this routine results in an UNLISTEN command being transmitted
; on the IEEE bus. Only devices previously commanded to listen will be affected.

; This routine is normally used after the computer is finished sending data to
; external devices. Sending the UNLISTEN will command the listening devices to get
; off the IEEE bus so it can be used for other purposes.

IEUNLSN
	LDA #$3F		; set the UNLISTEN command
	BIT DEFRFLG		; test deferred character flag
	BPL IECMD		; branch if no deferred character

	LDA VIABPCR		; get VIA B PCR
	AND #$DF		; set CB2 low, EOI low
	STA VIABPCR		; save VIA B PCR
	LDA DEFRCH		; get deferred byte
	JSR IESEND		; output a byte to the IEEE bus
	LDA VIABPCR		; get VIA B PCR
	ORA #$20		; set CB2 high, EOI high
	STA VIABPCR		; save VIA B PCR
	LSR DEFRFLG		; clear deferred character flag
	LDA #$3F		; set the UNLISTEN command

IECMD
	JSR LB33B		; send command
	JMP IECATN		; set ATN high


;***********************************************************************************;
;
; command the IEEE bus to UNTALK

; This routine will transmit an UNTALK command on the IEEE bus. All devices
; previously set to TALK will stop sending data when this command is received.

IEUNTLK
	LDA VIAAPB		; get VIA A DRB
	ORA #$06		; set NDAC out and NRFD out high
	STA VIAAPB		; set VIA A DRB
	LDA #$5F		; set the UNTALK command
	BNE IECMD		; send command and clear ATN, branch always


;***********************************************************************************;
;
; send secondary address after TALK

; This routine transmits a secondary address on the IEEE bus for a TALK device.
; This routine must be called with a number between 4 and 30 in the accumulator.
; The routine will send this number as a secondary address command over the serial
; bus. This routine can only be called after a call to the TALK routine. It will
; not work after a LISTEN.

; A secondary address is usually used to give set-up information to a device before
; I/O operations begin.

; When a secondary address is to be sent to a device on the IEEE bus the address
; must first be ORed with $60.

IETKSA
	JSR IESEND		; output a byte to the IEEE bus

IETKDONE
	LDA VIAAPB		; get VIA B DRB
	AND #$F9		; mask NRFD and NDAC out
	STA VIAAPB		; save VIA B DRB
	NOP			; short ..
	NOP
	NOP
	NOP
	NOP			; .. delay
	JMP IECATN		; set ATN high


;***********************************************************************************;
;
; input a byte from the IEEE bus

; This routine reads a byte of data from the IEEE bus using full handshaking. The
; data is returned in the accumulator. Before using this routine the TALK routine
; must have been called first to command the device on the IEEE bus to send data on
; the bus. If the input device needs a secondary command it must be sent before
; calling this routine.

; Errors are returned in the status word which can be read by calling the READST
; routine.

IEACPTR
.(
	LDA VIAAPB		; get VIA A DRB
	ORA #$02		; set NRFD out high
	STA VIAAPB		; set VIA A DRB
	LDA #$FF		; set for 65,280 cycles
	STA VIABT1CH		; set VIA B T1C_h
LABEL1	LDA VIAAPB		; get VIA A DRB
	AND #$10		; mask DAV in
	BEQ LABEL2		; branch if DAV low
	LDA VIABIFR		; get VIA B IFR
	AND #$40		; mask T1 interrupt
	BEQ LABEL1		; loop while timer not expired
	LDA #$02		; set time out read
	STA STATUS		; set I/O status byte
	RTS

LABEL2	LDA VIAAPB		; get VIA A DRB
	AND #$08		; mask EOI in
	BNE LABEL3		; branch if not EOI
	LDA #$40		; set EOI bit
	JSR ORIOST		; OR into I/O status byte
LABEL3	LDA VIABPB		; get VIA B DRB
	EOR #$FF		; invert it
	PHA			; save byte received
	LDA VIAAPB		; get VIA A DRB
	AND #$FD		; mask NRFD out
	STA VIAAPB		; store VIA A DRB
	LDA VIAAPB		; get VIA A DRB
	ORA #$04		; set NDAC out
	STA VIAAPB		; store VIA A DRB
LABEL4	LDA VIAAPB		; get VIA A DRB
	AND #$10		; mask DAV in
	BEQ LABEL4		; loop if DAV low
	LDA VIAAPB		; get VIA A DRB
	AND #$F9		; mask NRFD and NDAC out
	STA VIAAPB		; store VIA A DRB
	PLA			; restore byte received
	CLC			; flag ok
	RTS
.)


;************************************************************************************
;
; output a byte to the IEEE bus

; This routine is used to send information to devices on the IEEE bus. A call to
; this routine will put a data byte onto the IEEE bus using full handshaking.
; Before this routine is called the LISTEN routine must be used to command a device
; on the IEEE bus to get ready to receive data.

; The accumulator is loaded with a byte to output as data on the IEEE bus. A
; device must be listening or the status word will return a timeout. This routine
; always buffers one character. So when a call to the UNLSN routine is made to end
; the data transmission, the buffered character is sent with EOI set. Then the
; UNLISTEN command is sent to the device.

IECIOUT
.(
	PHA			; save byte to output
	LDA DEFRFLG		; get deferred character flag
	BPL LABEL1		; branch if no deferred character
	LDA DEFRCH		; get deferred byte
	JSR IESEND		; Tx byte on either serial or IEEE bus
LABEL1	SEC			; set carry
	ROR DEFRFLG		; shift into deferred character flag
	PLA			; restore byte
	STA DEFRCH		; save deferred byte
	CLC			; flag ok
	RTS
.)


;***********************************************************************************;
;
; open IEEE bus logical channel

IEOPNCHN
.(
	LDA SA			; get secondary address
	BMI LABEL1		; ok exit if no address

	LDY FNLEN		; get file name length
	BEQ LABEL1		; ok exit if null

	LDA FA			; get device number
	JSR IELISTEN		; command devices on the IEEE bus to LISTEN
	LDA SA			; get the secondary address
	ORA #$F0		; OR with the OPEN command
	JSR IECMD		; send secondary address after LISTEN
	LDA STATUS		; get I/O status byte
	BPL LABEL2		; branch if device present

	PLA			; else dump calling address low byte
	PLA			; dump calling address high byte
	JMP FE_DVNTP		; do device not present error and return

LABEL2
	LDA FNLEN		; get file name length
	BEQ LABEL3		; branch if null name

	LDY #$00		; clear index
LABEL4
	LDA (FNADR),Y		; get file name byte
	JSR IECIOUT		; output a byte to the IEEE bus
	INY			; increment index
	CPY FNLEN		; compare with file name length
	BNE LABEL4		; loop if not all done

LABEL3
	JSR IEUNLSN		; command the IEEE bus to UNLISTEN
LABEL1
	CLC			; flag ok
	RTS
.)


;***********************************************************************************;
;
; close IEEE bus logical channel

IECLOCHN
.(
	BIT SA			; test the secondary address
	BPL LABEL1
	RTS			; if already closed just exit

LABEL1	LDA FA			; get device number
	JSR IELISTEN		; command devices on the IEEE bus to LISTEN
	LDA SA			; get secondary address
	AND #$EF		; mask the channel number
	ORA #$E0		; OR with the CLOSE command
	JSR IECMD		; send IEEE secondary command
	JMP IEUNLSN		; command the IEEE bus to UNLISTEN
.)


;***********************************************************************************;
;***********************************************************************************;
;
; XKERNAL - Multi-bus KERNAL extension

	; KERNAL entrypoints
FTALK	= $EE14			; command a serial bus device to TALK
FLISTEN	= $EE17			; command devices on the serial bus to LISTEN
FSECOND	= $EEC0			; send secondary address after LISTEN
SCATN	= $EEC5			; set serial ATN high
FTKSA	= $EECE			; send secondary address after TALK
FCIOUT	= $EEE4			; output a byte to the serial bus
FUNTLK	= $EEF6			; command the serial bus to UNTALK
FUNLSN	= $EF04			; command the serial bus to UNLISTEN
FACPTR	= $EF19			; input a byte from the serial bus
FGETIN	= $F1F5			; get character from keyboard queue
FCHRIN	= $F20E			; input character from channel
FCHROUT	= $F27A			; output a character to channel
FLOAD2	= $F549
FSAVE2	= $F685
SAVING	= $F728			; print "SAVING <file name>"
FE_2MNYF = $F77E		; display too many files
FE_ALOPN = $F781		; display file already open
FE_NTFND = $F787		; display file not found
FE_DVNTP = $F78A		; display device not present
FE_NTINP = $F78D		; display not input file
FE_MISFN = $F793		; display missing file name
RD300	= $FBD2			; copy I/O start address to buffer address
VPRTY	= $FD11			; check read/write pointer
WRT62	= $FD1B			; increment read/write pointer


;***********************************************************************************;
;
; private variables

BUSMASK	= $92			; device address to bus mapping

BUSFLGS	= $0336			; bus to be used for I/O, b7 - output, b6 - input,
				;			0 = serial, 1 = IEEE


;***********************************************************************************;
;
; set device address to bus mapping
;
; Define which device address range gets mapped to the serial bus and which to the IEEE bus.
;
; .A = $00 - devices $04 - $1E on serial bus
;            devices $44 - $5E on IEEE bus
;      $40 - devices $04 - $1E on IEEE bus
;            devices $44 - $5E on serial bus

SETBUS
	STA BUSMASK
	RTS


;***********************************************************************************;
;
; get bus for a device address as defined by the bus mapping
;
; Vb = 0 - device on serial bus (or internal)
;      1 - device on IEEE bus

GETBUS
	PHA			; save device address
	EOR BUSMASK		; toggle top bit of pseudo device
	CLC
	ADC #$40-4		; if bus is IEEE and device >=4 then set Vb
	PLA			; restore device address
	AND #$BF		; mask top bit of pseudo device
	RTS


;***********************************************************************************;
;
; open logical channel

XOPNCHN
.(
	LDA FA			; get device number
	STA DSFA		; set disk status device number
	JSR GETBUS		; lookup bus mapping
	STA FA			; save masked device number
	BVS LABEL1		; if IEEE bus use IEEE routine
	JMP SERNAME		; use KERNAL routine for serial bus devices

LABEL1
	JMP IEOPNCHN		; open IEEE bus logical channel
.)


;***********************************************************************************;
;
; open a logical file

; This routine is used to open a logical file. Once the logical file is set up it
; can be used for input/output operations. Most of the I/O KERNAL routines call on
; this routine to create the logical files to operate on. No arguments need to be
; set up to use this routine, but both the SETLFS and SETNAM KERNAL routines must
; be called before using this routine.

XOPEN
.(
	LDX LA			; get logical file
	BNE LABEL1		; branch if there is a file

	JMP FE_NTINP		; else do not input file error and return

LABEL1
	JSR FNDFLNO		; find file
	BNE LABEL2		; branch if file not found

	JMP FE_ALOPN		; else do file already open error and return

LABEL2
	LDX LDTND		; get open file count
	CPX #$0A		; compare with max
	BCC LABEL3		; branch if less

	JMP FE_2MNYF		; else do too many files error and return

LABEL3
	INC LDTND		; increment open file count
	LDA LA			; get logical file
	STA LAT,X		; save to logical file table
	LDA SA			; get secondary address
	ORA #$60		; OR with the OPEN CHANNEL command
	STA SA			; set secondary address
	STA SAT,X		; save to secondary address table
	LDA FA			; get device number
	STA FAT,X		; save to device number table
	STA DSFA		; set disk status device number
	JSR GETBUS		; lookup bus mapping
	STA FA			; save masked device number
	BVS LABEL4		; if IEEE bus use IEEE routine
	JMP $F437		; use KERNAL routine for internal and serial bus devices

LABEL4
	JMP IEOPNCHN		; open IEEE bus logical channel
.)


;***********************************************************************************;
;
; close a specified logical file

; This routine is used to close a logical file after all I/O operations have been
; completed on that file. This routine is called after the accumulator is loaded
; with the logical file number to be closed, the same number used when the file was
; opened using the XOPEN routine.

XCLOSE
.(
	JSR $F3D4		; find file .A
	BEQ LABEL1		; if the file is found go close it

	CLC			; else the file was closed so just flag ok
	RTS

; found the file so close it

LABEL1
	JSR SETFLCH		; set file details from table,.X
	TXA			; copy file index to .A
	PHA			; save file index
	LDA FA			; get device number
	JSR GETBUS		; lookup bus mapping
	STA FA			; save masked device number
	BVS LABEL2		; if IEEE bus use IEEE routine
	JMP $F358		; use KERNAL routine for internal and serial bus devices

LABEL2
	JSR IECLOCHN		; close IEEE bus logical channel
	JMP $F3B1		; restore file index and close file
.)


;***********************************************************************************;
;
; open a channel for input

; Any logical file that has already been opened by the XOPEN routine can be defined as
; an input channel by this routine. The device on the channel must be an input device
; or an error will occur and the routine will abort.

; If you are getting data from anywhere other than the keyboard, this routine must be
; called before using either the XCHRIN routine or the XGETIN routine. If you are
; getting data from the keyboard and no other input channels are open then the calls
; to this routine and to the XOPEN routine are not needed.

; When used with a device on an external bus this routine will automatically send the
; listen address specified by the XOPEN routine and any secondary address.

; Possible errors are:
;
;	3 : file not open
;	5 : device not present
;	6 : file is not an input file

XCHKIN
.(
	JSR FNDFLNO		; find file
	BEQ LABEL1		; branch if file opened

	JMP FE_NTOPN		; do file not open error and return

LABEL1
	JSR SETFLCH		; set file details from table,.X
	LDA FA			; get device number
	JSR GETBUS		; lookup bus mapping
	STA FA			; save masked device number
	BVS LABEL2		; if IEEE bus use IEEE routine
	JSR $F2D4		; use KERNAL routine for internal and serial bus devices
	BCS LABEL3		; if error then return
	LDA BUSFLGS		; get bus flags
	AND #%10111111		; clear external bus input flag
LABEL4	STA BUSFLGS		; set bus flags
LABEL3	RTS

LABEL2
	TAX			; copy device number to .X
	JSR IETALK		; command an IEEE bus device to TALK
	LDA SA			; get secondary address
	BPL LABEL5		; branch if address to send

	JSR IETKDONE		; complete handshake and clear ATN
	JMP LABEL6		; do I/O status test

LABEL5
	JSR IETKSA		; send secondary address after TALK

LABEL6
	TXA			; copy device back to .A
	BIT STATUS		; test I/O status byte
	BPL LABEL7		; if device present save device number

	JMP FE_DVNTP		; do device not present error and return

LABEL7
	STA DFLTN		; save input device number
	CLC			; flag ok
	LDA BUSFLGS		; get bus flags
	ORA #%01000000		; set external bus input flag
	BNE LABEL4		; branch always
.)


;***********************************************************************************;
;
; input character from channel

; This routine will get a byte of data from the channel already set up as the input
; channel by the XCHKIN routine.

; If XCHKIN has not been used to define another input channel the data is expected to be
; from the keyboard. the data byte is returned in the accumulator. The channel remains
; open after the call.

; Input from the keyboard is handled in a special way. First, the cursor is turned on
; and it will blink until a carriage return is typed on the keyboard. All characters
; on the logical line, up to 88 characters, will be stored in the BASIC input buffer.
; Then the characters can be returned one at a time by calling this routine once for
; each character. When the carriage return is returned the entire line has been
; processed. the next time this routine is called the whole process begins again.

XCHRIN
	BIT BUSFLGS		; test bus flags
	BVS XIEIN		; branch if input from IEEE bus
	JMP FCHRIN		; use KERNAL routine for internal and serial bus devices

XIEIN
	JMP IEACPTR		; input a byte from the IEEE bus


;***********************************************************************************;
;
; get a character from the input device

; In practice this routine operates identically to the CHRIN routine for all devices
; except for the keyboard. If the keyboard is the current input device this routine
; will get one character from the keyboard buffer. It depends on the IRQ routine to
; read the keyboard and put characters into the buffer.

; If the keyboard buffer is empty the value returned in the accumulator will be zero.

XGETIN
	BIT BUSFLGS		; test bus flags
	BVS XIEIN		; if IEEE bus use IEEE routine
	JMP FGETIN		; use KERNAL routine for internal and serial bus devices


;***********************************************************************************;
;
; open a channel for output

; Any logical file that has already been opened by the XOPEN routine can be defined
; as an output channel by this routine the device on the channel must be an output
; output device or an error will occur and the routine will abort.

; If you are sending data to anywhere other than the screen this routine must be
; called before using the XCHROUT routine. If you are sending data to the screen and
; no other output channels are open then the calls to this routine and to the XOPEN
; routine are not needed.

; When used with a device on an external bus this routine will automatically send the
; listen address specified by the XOPEN routine and any secondary address.

; Possible errors are:
;
;	3 : file not open
;	5 : device not present
;	7 : file is not an output file

XCHKOUT
.(
	JSR FNDFLNO		; find file
	BEQ LABEL1		; branch if file found

	JMP FE_NTOPN		; do file not open error and return

LABEL1
	JSR SETFLCH		; set file details from table,.X
	LDA FA			; get device number
	JSR GETBUS		; lookup bus mapping
	BVS LABEL2		; if IEEE bus use IEEE routine
	STA FA			; save masked device number
	JSR $F316		; use KERNAL routine for internal and serial bus devices
	BCS LABEL3		; if error then return
	LDA BUSFLGS		; get bus flags
	AND #%01111111		; clear external bus output flag
LABEL4	STA BUSFLGS		; set bus flags
LABEL3	RTS

LABEL2
	TAX			; copy device number to .X
	JSR IELISTEN		; command devices on the IEEE bus to LISTEN
	LDA SA			; get secondary address
	BPL LABEL5		; branch if address to send

	JSR IECATN		; set ATN high
	JMP LABEL6		; do I/O status test

LABEL5
	JSR IECMD		; send secondary address after LISTEN

LABEL6
	TXA			; copy device back to .A
	BIT STATUS		; test I/O status byte
	BPL LABEL7		; if device present save device number

	JMP FE_DVNTP		; do device not present error and return

LABEL7
	STA DFLTO		; save output device number
	CLC			; flag ok
	LDA BUSFLGS		; get bus flags
	ORA #%10000000		; set external bus output flag
	BNE LABEL4		; branch always
.)


;***********************************************************************************;
;
; output a character to channel

; This routine will output a character to an already opened channel. Use the XOPEN
; routine and the XCHKOUT routine  to set up the output channel before calling
; this routine. If these calls are omitted, data will be sent to the default output
; device, device 3, the screen. The data byte to be output is loaded into the accumulator,
; and this routine is called. The data is then sent to the specified output device.
; The channel is left open after the call.

; NOTE: Care must be taken when using routine to send data to an external device since
; data will be sent to all open output channels on the bus. Unless this is desired,
; all open output channels on an external bus other than the actually intended
; destination channel must be closed by a call to the KERNAL close channel routine.

XCHROUT
.(
	BIT BUSFLGS		; test bus flags
	BMI LABEL1		; branch if output to IEEE bus
	JMP FCHROUT		; use KERNAL routine for internal and serial bus devices

LABEL1
	JMP IECIOUT		; output a byte to the IEEE bus
.)


;***********************************************************************************;
;
; close input and output channels

; This routine is called to clear all open channels and restore the I/O channels to
; their original default values. It is usually called after opening other I/O
; channels and using them for input/output operations. The default input device is
; 0, the keyboard. The default output device is 3, the screen.

; If one of the channels to be closed is on an external bus, an UNTALK signal is sent
; first to clear the input channel or an UNLISTEN is sent to clear the output channel.
; By not calling this routine and leaving listener(s) active on an external bus,
; several devices can receive the same data from the VIC at the same time. One way to
; take advantage of this would be to command the printer to LISTEN and the disk to
; TALK. This would allow direct printing of a disk file.

XCLRCHN
.(
	LDX #$03		; set .X to screen
	CPX DFLTO		; compare output device number with screen
	BCS LABEL1		; branch if >= screen

				; else was external bus
	JSR XUNLSN		; command an external bus to UNLISTEN
LABEL1
	CPX DFLTN		; compare input device number with screen
	BCS LABEL2		; branch if >= screen

				; else was external bus
	JSR XUNTLK		; command an external bus to UNTALK
LABEL2
	STX DFLTO		; set output device number to screen
	LDA #$00		; set for keyboard
	STA DFLTN		; set input device number to keyboard
	STA BUSFLGS		; clear bus flags
	RTS
.)


;***********************************************************************************;
;
; command an external bus device to TALK

; To use this routine the accumulator must first be loaded with a device number
; between 4 and 30. When called this routine converts this device number to a talk
; address. Then this data is transmitted as a command on the bus.

XTALK
.(
	STA DSFA		; set disk status device number
	JSR GETBUS		; lookup bus mapping
	BVS LABEL1		; branch if IEEE bus
	JSR FTALK		; command a serial bus device to TALK
	LDA STATUS		; get I/O status byte
	BNE LABEL2		; exit if any error
	LDA BUSFLGS		; get bus flags
	AND #%10111111		; clear external bus input flag
LABEL3	STA BUSFLGS		; set bus flags
LABEL2	RTS

LABEL1
	JSR IETALK		; command an IEEE bus device to TALK
	LDA STATUS		; get I/O status byte
	BNE LABEL2		; exit if any error
	LDA BUSFLGS		; get bus flags
	ORA #%01000000		; set external bus input flag
	BNE LABEL3		; branch always
.)


;***********************************************************************************;
;
; send secondary address after TALK

; This routine transmits a secondary address on an external bus for a TALK device.
; This routine must be called with a number between 4 and 30 in the accumulator.
; The routine will send this number as a secondary address command over the serial
; bus. This routine can only be called after a call to the XTALK routine. It will
; not work after a LISTEN.

; A secondary address is usually used to give set-up information to a device before
; I/O operations begin.

; When a secondary address is to be sent to a device on an external bus the address
; must first be ORed with $60.

XTKSA
.(
	BMI NOSA		; branch if no secondary address
	BIT BUSFLGS		; test bus flags
	BVS LABEL1		; branch if input from IEEE bus
	JMP FTKSA		; send secondary address after TALK

LABEL1	JMP IETKSA		; send secondary address after TALK

NOSA
	BIT BUSFLGS		; test bus flags
	BVS LABEL2		; branch if input from IEEE bus
	JMP $EED3		; wait for bus end after send

LABEL2	JMP IETKDONE		; complete handshake and clear ATN
.)


;***********************************************************************************;
;
; command an external bus to UNTALK

; This routine will transmit an UNTALK command on an external bus. All devices
; previously set to TALK will stop sending data when this command is received.

XUNTLK
.(
	BIT BUSFLGS		; test bus flags
	LDA BUSFLGS		; get bus flags
	AND #%10111111		; mask external bus input flag
	STA BUSFLGS		; set bus flags
	BVS LABEL1		; branch if input from IEEE bus
	JMP FUNTLK		; command the serial bus to UNTALK

LABEL1
	JMP IEUNTLK		; command the IEEE bus to UNTALK
.)


;***********************************************************************************;
;
; input a byte from an external bus

; This routine reads a byte of data from an external bus using full handshaking. The
; data is returned in the accumulator. Before using this routine the XTALK routine
; must have been called first to command the device on the external bus to send data on
; the bus. If the input device needs a secondary command it must be sent by using the
; XTKSA routine before calling this routine.

; Errors are returned in the status word which can be read by calling the READST
; routine.

XACPTR
.(
	BIT BUSFLGS		; test bus flags
	BVS LABEL1		; branch if input from IEEE bus
	JMP FACPTR		; input a byte from the serial bus

LABEL1
	JMP IEACPTR		; input a byte from the IEEE bus

.)


;***********************************************************************************;
;
; command devices on an external bus to LISTEN

; This routine will command a device on an external bus to receive data. The
; accumulator must be loaded with a device number between 4 and 30 before calling
; this routine. LISTEN convert this to a listen address then transmit this data as
; a command on the bus. The specified device will then go into listen mode
; and be ready to accept information.

XLISTEN
.(
	JSR GETBUS		; lookup bus mapping
	BVS LABEL1		; branch if IEEE bus
	JSR FLISTEN		; command devices on the serial bus to LISTEN
	LDA STATUS		; get I/O status byte
	BNE LABEL2		; if any error then return
	LDA BUSFLGS		; get bus flags
	AND #%01111111		; clear external bus output flag
LABEL3	STA BUSFLGS		; set bus flags
LABEL2	RTS

LABEL1
	JSR IELISTEN		; command devices on the IEEE bus to LISTEN
	LDA STATUS		; get I/O status byte
	BNE LABEL2		; if any error then return
	LDA BUSFLGS		; get bus flags
	ORA #%10000000		; set external bus output flag
	BNE LABEL3		; branch always
.)


;***********************************************************************************;
;
; send secondary address after LISTEN

; This routine is used to send a secondary address to an I/O device after a call to
; the XLISTEN routine is made and the device commanded to LISTEN. The routine cannot
; be used to send a secondary address after a call to the XTALK routine.

; A secondary address is usually used to give set-up information to a device before
; I/O operations begin.

; When a secondary address is to be sent to a device on an external bus the address
; must first be ORed with $60.

XSECOND
.(
	BMI NOSA		; branch if no secondary address
	BIT BUSFLGS		; test bus flags
	BMI LABEL1		; branch if ouput to IEEE bus
	JMP FSECOND		; send secondary address after LISTEN

LABEL1
	JMP IECMD		; send secondary address after LISTEN

NOSA
	BIT BUSFLGS		; test bus flags
	BMI LABEL2		; branch if ouput to IEEE bus
	JMP SCATN		; set serial ATN high

LABEL2
	JMP IECATN		; set ATN high
.)


;***********************************************************************************;
;
; command an external bus to UNLISTEN

; This routine commands all devices on an external bus to stop receiving data from
; the computer. Calling this routine results in an UNLISTEN command being transmitted
; on the bus. Only devices previously commanded to listen will be affected.

; This routine is normally used after the computer is finished sending data to
; external devices. Sending the UNLISTEN will command the listening devices to get
; off the bus so it can be used for other purposes.

XUNLSN
.(
	LDA BUSFLGS		; get bus flags
	PHP			; save flags, Nb will be cleared by next instruction
	AND #%01111111		; mask external bus output flag
	STA BUSFLGS		; set bus flags
	PLP			; restore flags
	BMI LABEL1		; branch if output from IEEE bus
	JMP FUNLSN		; command the serial bus to UNLISTEN

LABEL1
	JMP IEUNLSN		; command the IEEE bus to UNLISTEN
.)


;***********************************************************************************;
;
; output a byte to an external bus

; This routine is used to send information to devices on an external bus. A call to
; this routine will put a data byte onto the bus using full handshaking.
; Before this routine is called the XLISTEN routine must be used to command a device
; on the bus to get ready to receive data.

; The accumulator is loaded with a byte to output as data on the bus. A
; device must be listening or the status word will return a timeout. This routine
; always buffers one character. So when a call to the XUNLSN routine is made to end
; the data transmission, the buffered character is sent with EOI set. Then the
; UNLISTEN command is sent to the device.

XCIOUT
.(
	BIT BUSFLGS		; test bus flags
	BMI LABEL1		; branch if output to IEEE bus
	JMP FCIOUT		; output a byte to the serial bus

LABEL1
	JMP IECIOUT		; output a byte to the IEEE bus
.)


;***********************************************************************************;
;
; load RAM from a device

; This routine will load data bytes from any input device directly into the memory
; of the computer. It can also be used for a verify operation comparing data from a
; device with the data already in memory, leaving the data stored in RAM unchanged.

; The accumulator must be set to 0 for a load operation or 1 for a verify. If the
; input device was OPENed with a secondary address of 0 the header information from
; device will be ignored. In this case .X.Y must contain the starting address for the
; load. If the device was addressed with a secondary address of 1 or 2 the data will
; load into memory starting at the location specified by the header. This routine
; returns the address of the highest RAM location which was loaded.

; Before this routine can be called, the SETLFS and SETNAM routines must be called.

XLOAD
	STX MEMUSS		; set load start address low byte
	STY MEMUSS+1		; set load start address high byte

XLOAD2
.(
	PHA			; save load/verify flag
	LDY #$03		; set index to string length
LABELB	LDA KEYD,Y		; get character from keyboard buffer
	CMP RUNCR,Y		; compare with "RUN"
	BNE LABELA		; branch if no match
	DEY			; decrement index
	BPL LABELB		; loop until done
	LDA FNLEN		; get file name length
	BNE LABELA		; branch if not null name

	LDA #$08		; set device to 8, first disk
	STA FA			; set device number
	LDA #$03		; set default file name length
	STA FNLEN		; set file name length
	LDA #>DFLTNAME		; get default name high byte
	STA FNADR+1		; set file name pointer high byte
	LDA #<DFLTNAME		; get default name low byte
	STA FNADR		; set file name pointer low byte

LABELA	LDA FA			; get device number
	STA DSFA		; set disk status device number
	JSR GETBUS		; lookup bus mapping
	STA FA			; save masked device number
	PLA			; restore load/verify flag
	BVS LABEL1		; if IEEE bus use IEEE routine
	JMP FLOAD2

LABEL1
	STA VERCK		; save load/verify flag
	LDA #$00		; clear .A
	STA STATUS		; clear I/O status byte
	LDY FNLEN		; get file name length
	BNE LABEL2		; branch if not null name

	JMP FE_MISFN		; else do missing file name error and return

LABEL2
	JSR PATCH1		; get secondary address and print "SEARCHING..."
	LDA #$60		; set secondary address to $00
	STA SA			; save secondary address
	JSR IEOPNCHN		; open IEEE bus logical channel
	LDA FA			; get device number
	JSR IETALK		; command an IEEE bus device to TALK
	LDA SA			; get secondary address
	JSR IETKSA		; send secondary address after TALK
	JSR IEACPTR		; input a byte from the IEEE bus
	STA EAL			; save program start address low byte
	LDA STATUS		; get I/O status byte
	LSR			; shift time out read ..
	LSR			; .. into carry bit
	BCS LABEL3		; if timed out go do file not found error and return

	JSR IEACPTR		; input a byte from the IEEE bus
	STA EAL+1		; save program start address high byte
	JSR PATCH2		; set LOAD address if secondary address = 0
LABEL4
	LDA #$FD		; mask xxxx xx0x, clear time out read bit
	AND STATUS		; mask I/O status byte
	STA STATUS		; set I/O status byte
	JSR STOP		; scan stop key, return Zb = 1 = [STOP]
	BNE LABEL5		; branch if not [STOP]

	JMP CLOSTOP		; else close the IEEE bus device and flag stop

LABEL3
	JSR IEUNTLK		; command the IEEE bus to UNTALK
	JSR IECLOCHN		; close IEEE bus logical channel
	JMP FE_NTFND		; do file not found error and return

LABEL5
	JSR IEACPTR		; input a byte from the IEEE bus
	TAX			; copy byte
	LDA STATUS		; get I/O status byte
	LSR			; shift time out read ..
	LSR			; .. into carry bit
	BCS LABEL4		; if timed out clear I/O status and retry

	TXA			; copy received byte back
	LDY VERCK		; get load/verify flag
	BEQ LABEL6		; branch if load

				; else is verify
	LDY #$00		; clear index
	CMP (EAL),Y		; compare byte with previously loaded byte
	BEQ LABEL7		; branch if match

	LDA #$10		; set read error bit
	JSR ORIOST		; OR into I/O status byte
	.BYTE $2C		; makes next line BIT $AE91
LABEL6
	STA (EAL),Y		; save byte to memory
LABEL7
	INC EAL			; increment save pointer low byte
	BNE LABEL8		; if no rollover skip the high byte increment

	INC EAL+1		; else increment save pointer high byte
LABEL8
	BIT STATUS		; test I/O status byte
	BVC LABEL4		; loop if not end of file

	JSR IEUNTLK		; command the IEEE bus to UNTALK
	JSR IECLOCHN		; close IEEE bus logical channel
	CLC			; flag ok
	LDX EAL			; get the LOAD end pointer low byte
	LDY EAL+1		; get the LOAD end pointer high byte
	RTS
.)


;***********************************************************************************;
;
; save RAM to device, .A = index to start address, .X.Y = end address low/high

; This routine saves a section of memory. Memory is saved from an indirect address
; on page 0 specified by A, to the address stored in .X.Y, to a logical file. The
; SETLFS and SETNAM routines must be used before calling this routine. However, a
; file name is not required to SAVE to device 1, the cassette. Any attempt to save to
; other devices without using a file name results in an error.

; NOTE: device 0, the keyboard, and device 3, the screen, cannot be SAVEd to. If
; the attempt is made, an error will occur, and the SAVE stopped.

XSAVE
	STX EAL			; save end address low byte
	STY EAL+1		; save end address high byte
	TAX			; copy index to start pointer
	LDA $00,X		; get start address low byte
	STA STAL		; set I/O start address low byte
	LDA $01,X		; get start address high byte
	STA STAL+1		; set I/O start address high byte

XSAVE2
.(
	LDA FA			; get device number
	STA DSFA		; set disk status device number
	JSR GETBUS		; lookup bus mapping
	STA FA			; save masked device number
	BVS LABEL1		; if IEEE bus use IEEE routine
	JMP FSAVE2

LABEL1
	LDA #$61		; set secondary address to $01
				; when a secondary address is to be sent to a device on
				; an external bus the address must first be ORed with $60
	STA SA			; save secondary address
	LDY FNLEN		; get file name length
	BNE LABEL2		; branch if filename not null

	JMP FE_MISFN		; else do missing file name error and return

LABEL2
	JSR IEOPNCHN		; open IEEE bus logical channel
	JSR SAVING		; print "SAVING <file name>"
	LDA FA			; get device number
	JSR IELISTEN		; command devices on the IEEE bus to LISTEN
	LDA SA			; get secondary address
	JSR IECMD		; send secondary address after LISTEN
	LDY #$00		; clear index
	JSR RD300		; copy I/O start address to buffer address
	LDA SAL			; get buffer address low byte
	JSR IECIOUT		; output a byte to the IEEE bus
	LDA SAL+1		; get buffer address high byte
	JSR IECIOUT		; output a byte to the IEEE bus
LABEL3
	JSR VPRTY		; check read/write pointer, return Cb = 1 if pointer >= end
	BCS LABEL4		; go do UNLISTEN if at end

	LDA (SAL),Y		; get byte from buffer
	JSR IECIOUT		; output a byte to the IEEE bus
	JSR STOP		; scan stop key
	BEQ CLOSTOP		; if stop close and flag stop

	JSR WRT62		; increment read/write pointer
	BNE LABEL3		; loop, branch always

LABEL4
	JSR IEUNLSN		; command the IEEE bus to UNLISTEN
	JSR IECLOCHN		; close IEEE bus logical channel
	CLC
	RTS
.)

; close the IEEE bus device and flag stop

CLOSTOP
	JSR IECLOCHN		; close IEEE bus logical channel
	LDA #ER_STOP		; terminated by [STOP] key
	SEC			; flag stop
	RTS
